/**
  @file assets.h

  This file containts sounds and music that can optionally be used by the game
  frontend. Every sound effect has 2048 samples, is stored as 8kHz mono format
  with 4 bit quantization, meaning every sound effect takes 1024 bytes. Sounds
  can be converted using a provided python script like this:

  python snd2array.py sound.raw

  Music is based on bytebeat (procedural waveforms generated by short bitwise
  operation formulas). The formulas were NOT copied from anywhere, they were
  discovered from scratch.

  by Miloslav Ciz (drummyfish), 2019

  Released under CC0 1.0 (https://creativecommons.org/publicdomain/zero/1.0/)
  plus a waiver of all other intellectual property. The goal of this work is to
  be and remain completely in the public domain forever, available for any use
  whatsoever.
*/

#ifndef _SFG_SOUNDS_H
#define _SFG_SOUNDS_H

#define SFG_SFX_SAMPLE_COUNT 2048
#define SFG_SFX_SIZE (SFG_SFX_SAMPLE_COUNT / 2)

/**
  Gets an 8bit sound sample.
*/
#define SFG_GET_SFX_SAMPLE(soundIndex,sampleIndex) \
  ((SFG_PROGRAM_MEMORY_U8(SFG_sounds + soundIndex * SFG_SFX_SIZE \
  + sampleIndex / 2) << (4 * ((sampleIndex % 2) != 0))) & 0xf0)

#define SFG_TRACK_SAMPLES (512 * 1024)
#define SFG_TRACK_COUNT 6

/**
  Average value of each music track, can be used to correct DC offset issues if
  they appear.
*/
SFG_PROGRAM_MEMORY uint8_t SFG_musicTrackAverages[SFG_TRACK_COUNT] =
  {14,7,248,148,6,8};

struct
{ // all should be initialized to 0 by default
  uint8_t track;
  uint32_t t;      // time variable/parameter
  uint32_t t2;     // stores t squared, for better performance
  uint32_t n11t;   // stores a multiple of 11, for better performance
} SFG_MusicState;

/**
  Gets the next 8bit 8KHz music sample for the bytebeat soundtrack. This
  function is to be used by the frontend that plays music.
*/
uint8_t SFG_getNextMusicSample(void)
{
  if (SFG_MusicState.t >= SFG_TRACK_SAMPLES)
  {
    SFG_MusicState.track++;

    if (SFG_MusicState.track >= SFG_TRACK_COUNT)
      SFG_MusicState.track = 0;

    SFG_MusicState.t = 0;
    SFG_MusicState.t2 = 0;
    SFG_MusicState.n11t = 0;
  }

  uint32_t result;

  #define S SFG_MusicState.t // can't use "T" because of a C++ template
  #define S2 SFG_MusicState.t2
  #define N11S SFG_MusicState.n11t

  /* CAREFUL! Bit shifts in any direction by amount greater than data type
     width (32) are undefined behavior. Use % 32. */

  switch (SFG_MusicState.track) // individual music tracks
  {
    case 0:
    {
      uint32_t a = ((S >> 7) | (S >> 9) | (~S << 1) | S);
      result = (((S) & 65536) ? (a & (((S2) >> 16) & 0x09)) : ~a);

      SFG_MusicState.t2 += S;

      break;
    }

    case 1:
    {
      uint32_t a = (S >> 10);
      result = S & (3 << (((a ^ (a << ((S >> 6) % 32)))) % 32));

      break;
    }

    case 2:
    {
      result = 
        ~((((S >> ((S >> 2) % 32)) | (S >> ((S >> 5) % 32))) & 0x12) << 1) 
        | (S >> 11);

      break;
    }

    case 3:
    {
      result =
        (((((S >> ((S >> 2) % 32)) + (S >> ((S >> 7) % 32)))) & 0x3f) | (S >> 5)
        | (S >> 11)) & ((S & (32768 | 8192)) ? 0xf0 : 0x30);

      break;
    }

    case 4:
    {
      result =
        ((0x47 >> ((S >> 9) % 32)) & (S >> (S % 32))) | 
        (0x57 >> ((S >> 7) % 32)) |
        (0x06 >> ((S >> ((((N11S) >> 14) & 0x0e) % 32)) % 32));

      SFG_MusicState.n11t += 11;

      break;
    }

    case 5:
    {
      uint32_t a = S >> ((S >> 6) % 32);
      uint32_t b = 0x011121 >> (((a + S) >> 11) % 32);
      result = 
        (((S >> 9) + (S ^ (S << 1))) & (0x7f >> (((S >> 15) & 0x03) % 32))) 
        & (b + a);

      break;
    }

    default:
      result = 127;
      break;
  }

  #undef S
  #undef S2
  #undef N11S

  SFG_MusicState.t += 1;

  return result;
}

/**
  Switches the bytebeat to next music track.
*/
void SFG_nextMusicTrack(void)
{
  uint8_t current = SFG_MusicState.track;

  while (SFG_MusicState.track == current)
    SFG_getNextMusicSample();
}

SFG_PROGRAM_MEMORY uint8_t SFG_sounds[SFG_SFX_SIZE * 6] =
{
// 0, bullet shot
135,119,120,136,136,153,153,153,154,169,152,119,101,85,86,102,119,118,119,
85,84,51,33,52,52,84,87,120,170,188,202,152,102,84,84,70,119,136,119,
119,121,154,219,170,137,117,82,18,36,34,33,20,67,68,70,137,172,189,237,
220,150,120,120,97,36,102,121,151,87,169,118,86,102,120,137,135,120,186,155,
223,255,217,103,100,70,119,118,84,34,36,122,204,220,168,138,170,170,223,199,
117,70,119,136,100,85,102,51,37,101,103,118,101,136,87,154,169,171,187,186,
169,153,136,117,68,84,66,18,19,50,52,51,102,121,139,186,169,171,186,152,
153,136,119,134,85,101,86,69,84,84,86,85,86,102,119,120,153,135,135,101,
87,134,103,135,101,103,119,135,152,120,136,135,137,136,151,134,87,119,136,119,
118,102,85,119,85,102,102,119,138,137,153,137,186,170,137,152,135,101,85,85,
86,102,102,119,119,102,103,119,137,152,138,153,154,169,153,152,137,151,118,85,
85,84,84,86,86,136,119,119,154,153,153,171,187,170,170,187,170,137,151,119,
102,103,69,102,118,120,120,138,153,169,170,169,153,135,119,119,102,118,105,136,
136,137,152,153,136,152,119,119,119,119,121,152,136,119,152,136,135,120,119,118,
86,102,103,136,135,137,153,136,152,119,119,118,102,86,85,102,102,102,102,120,
136,136,136,136,152,136,153,152,119,119,120,135,120,119,119,103,119,136,119,135,
120,135,136,136,137,153,153,152,154,152,153,137,152,136,135,119,136,136,136,153,
152,154,170,170,153,153,152,119,119,119,119,118,119,103,136,136,120,135,118,120,
119,118,102,119,102,102,103,119,118,103,102,102,119,135,119,119,119,119,119,119,
119,118,102,103,135,136,135,119,120,135,119,119,119,119,103,119,120,136,137,152,
136,136,136,153,153,136,153,153,153,153,153,152,153,136,136,135,119,135,119,119,
136,136,136,136,152,152,137,153,152,119,118,102,102,102,119,103,119,119,119,136,
136,135,118,103,119,120,136,136,136,136,136,136,136,119,118,102,119,119,119,136,
136,136,136,137,136,136,136,136,119,119,120,135,119,119,120,135,136,136,136,136,
136,136,119,119,120,119,120,136,136,135,119,120,119,119,119,119,119,120,136,152,
136,137,153,136,136,136,136,136,136,136,119,120,137,153,136,136,135,119,119,136,
136,136,135,119,119,102,119,120,135,119,119,119,136,136,136,118,102,103,119,136,
119,119,120,136,136,136,135,119,119,136,136,136,136,136,136,136,136,135,119,119,
119,119,119,136,119,119,119,136,136,136,136,135,120,136,136,136,119,119,119,120,
136,136,136,136,135,119,119,119,119,136,119,119,136,136,136,136,135,119,119,119,
119,119,119,119,119,136,136,136,136,136,135,119,119,119,119,119,119,119,136,136,
136,136,135,120,136,136,136,119,119,119,136,136,136,135,119,119,119,119,119,119,
119,119,119,119,136,136,120,136,136,136,136,136,119,119,120,136,136,136,119,119,
120,136,136,136,136,136,136,136,136,136,136,136,135,119,119,119,119,119,119,119,
120,136,136,136,135,119,119,119,119,136,136,136,136,136,135,119,119,119,119,119,
119,120,136,136,136,136,136,135,119,119,119,119,119,119,119,120,136,136,136,136,
136,136,136,136,136,136,119,119,119,119,119,119,119,119,119,119,136,136,136,136,
136,136,136,136,136,136,136,119,119,119,119,119,119,119,119,136,136,136,136,136,
136,136,136,136,136,136,136,119,119,119,119,119,119,119,119,119,119,136,136,136,
136,136,136,136,119,119,119,119,119,120,136,136,136,136,136,136,136,135,119,119,
136,136,119,119,119,119,119,119,120,135,120,136,136,136,136,136,136,136,136,135,
119,119,119,119,119,119,119,119,136,136,136,136,136,136,136,136,136,135,119,119,
119,119,119,119,119,119,119,119,119,119,136,136,136,136,136,136,136,136,119,119,
119,119,119,119,119,120,136,136,136,136,136,136,136,119,119,119,119,119,119,119,
119,119,119,136,135,119,120,119,119,120,136,136,136,136,136,136,119,119,119,119,
119,119,119,119,120,136,136,136,136,136,136,136,119,119,135,119,119,119,119,119,
119,119,119,119,135,120,136,136,136,136,136,135,119,119,119,119,119,120,119,119,
119,135,119,136,136,136,136,136,136,135,119,119,119,119,119,119,119,119,136,136,
136,136,136,136,136,136,135,119,136,136,135,119,119,119,119,119,119,119,119,119,
119,119,136,136,136,136,136,136,136,136,136,119,119,119,119,119,119,119,136,136,
136,136,136,136,136,136,135,119,119,135,135,120,120,120,120,120,120,120,120,135,
135,136,120,120,135
, // 1, door opening
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,136,
136,136,136,136,136,136,136,136,136,136,153,153,153,153,153,153,153,153,153,153,
153,153,152,136,136,136,136,136,136,136,136,136,119,119,119,119,119,119,119,119,
119,119,119,102,102,102,102,103,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,136,136,136,136,136,136,136,136,136,136,153,153,153,153,153,
153,153,136,136,136,136,136,136,135,119,119,119,119,119,119,119,119,119,102,102,
102,102,102,102,102,102,102,103,119,119,119,119,120,136,136,136,136,136,137,153,
153,153,153,153,152,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
135,119,119,119,119,120,136,136,136,136,136,136,136,137,136,136,136,136,136,136,
136,136,136,136,136,136,135,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,119,119,119,136,136,136,136,136,136,120,
136,136,136,136,136,136,136,135,120,136,136,135,119,120,135,119,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,120,136,136,136,136,136,119,120,136,136,
136,136,136,135,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,120,136,136,136,136,136,136,136,136,136,136,136,
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,
136,136,136,136,136,153,153,153,153,153,152,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,135,120,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,135,119,119,119,119,119,119,119,119,119,119,120,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,119,119,119,119,119,119,119,119,119,136,136,136,136,136,136,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,102,102,102,102,102,103,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,120,136,136,136,136,119,119,119,119,119,119,119,119,
119,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,135,119,119,119,119,119,119,119,119,119,119,119,119,
119,120,136,136,136,136,136,136,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,120,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,120,135,119,119,119,136,136,136,136,
136,136,135,136,136,136,136,136,136,136,135,136,136,136,136,120,119,135,119,119,
119,119,119,119,119,120,136,136,120,136,136,136,136,136,119,136,135,136,136,136,
136,136,136,136,119,119,120,135,119,135,119,136,135,119,120,120,136,136,136,136,
135,119,119,119,119,119,119,119,119,120,119,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,119,119,119,103,119,119,119,102,102,102,102,102,102,118,103,
102,118,103,136,136,136,136,136,136,136,136,136,137,153,153,153,170,169,153,153,
170,153,153,153,170,170,169,170,169,153,153,153,153,153,153,170,170,170,153,153,
153,136,137,153,136,136,137,152,119,102,120,136,136,135,119,119,119,120,135,119,
119,120,137,153,153,153,152,136,135,119,119,119,102,119,119,119,119,119,119,119,
120,136,136,119,120,137,152,137,136,136,136,136,119,120,135,119,118,102,102,102,
102,102,102,119,119,119,119,118,103,119,119,119,119,119,102,102,102,85,85,85,
85,84,85,85,85,86,102,102,102,102,102,101,85,86,102,102,102,102,102,102,
102,102,102,119,102,102,119,119,119,120,136,119,119,119,120,136,136,136,136,136,
136,135,119,119,136,136,136,136,136,136,119,120,135,119,119,119,119,119,119,119,
119,119,119,120,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,120,120
, // 2, explosion 
135,136,153,18,51,51,33,18,123,255,255,255,255,255,255,255,254,184,48,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,55,189,255,255,255,
255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,255,254,201,135,
101,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,0,41,
239,255,255,253,186,188,221,221,220,153,152,136,155,188,203,187,171,202,169,116,
35,16,0,0,17,20,68,87,191,255,255,255,253,221,221,202,115,16,0,0,
0,0,18,34,70,117,85,68,85,86,102,68,67,68,70,136,153,134,67,32,
0,0,0,0,35,87,154,205,238,255,255,255,255,255,255,255,255,255,255,255,
255,255,237,168,101,67,16,0,0,0,53,102,119,133,85,85,49,0,0,34,
34,16,1,35,69,103,119,101,86,103,102,120,119,102,137,206,255,238,238,202,
152,120,134,85,86,102,102,102,119,120,135,117,68,50,34,35,69,121,188,221,
222,239,255,255,255,255,220,204,186,153,153,135,102,137,153,151,100,51,51,35,
69,102,68,68,67,52,68,51,86,118,86,119,118,103,137,172,221,237,221,221,
221,220,169,136,118,84,68,68,68,69,121,189,237,220,203,186,170,152,119,119,
120,170,188,204,204,204,188,204,186,152,117,67,50,52,87,119,118,103,102,103,
136,101,50,33,1,34,34,35,69,86,120,136,153,153,153,152,135,100,67,51,
51,69,85,102,121,188,205,222,255,236,203,204,187,188,221,203,170,170,170,169,
152,118,102,86,102,119,136,137,169,153,169,152,135,119,101,51,34,51,68,85,
102,85,85,84,85,102,102,85,86,103,137,170,187,221,204,222,255,238,237,203,
170,171,186,152,119,120,136,136,136,135,119,120,119,138,187,185,152,119,119,136,
134,83,16,1,35,68,68,50,17,52,104,172,222,238,238,238,221,220,186,153,
133,51,68,50,18,69,65,1,89,207,255,255,255,255,255,255,255,255,255,255,
255,255,255,255,255,255,255,252,184,81,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,2,53,104,154,223,255,255,255,255,255,255,255,255,
255,255,255,255,255,255,255,255,255,255,255,255,237,186,152,118,84,49,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,19,70,121,171,205,255,255,
255,255,255,255,255,255,255,254,220,187,170,153,152,135,102,118,101,67,16,0,
0,0,0,0,0,1,52,86,137,153,135,103,102,67,33,17,35,53,102,103,
118,102,84,51,35,51,51,69,87,120,154,189,255,255,255,255,255,255,255,236,
185,118,84,50,33,18,17,34,34,17,17,16,0,0,0,0,0,18,52,68,
69,86,119,137,171,187,205,221,237,239,255,255,255,253,204,186,152,136,118,66,
16,18,35,52,85,68,68,86,119,119,119,120,136,135,120,136,136,136,119,101,
85,68,68,67,50,17,16,0,0,0,1,17,17,17,18,35,69,120,171,188,
222,237,221,239,255,239,255,255,255,255,238,238,238,238,236,185,153,153,152,118,
85,84,67,51,50,34,34,34,35,51,34,34,35,52,68,68,68,69,85,103,
136,136,136,154,171,204,205,222,238,255,255,255,255,255,255,254,237,203,186,153,
153,153,153,136,135,118,102,84,50,16,0,0,0,0,0,0,0,0,0,17,
17,17,17,17,35,69,103,137,171,204,222,255,255,255,255,255,255,238,238,220,
203,170,169,153,170,170,171,187,187,205,221,220,203,186,136,135,119,102,85,68,
68,68,68,68,50,34,17,0,0,0,0,0,0,19,85,119,136,154,187,204,
204,186,136,120,136,136,136,136,136,154,188,239,255,255,255,255,238,237,203,186,
153,135,119,118,102,102,102,102,103,136,153,152,135,118,101,68,68,67,50,17,
0,0,0,0,1,17,17,34,51,52,86,103,137,170,170,187,187,204,221,221,
204,203,170,170,171,187,170,170,153,153,153,153,153,153,153,153,136,119,119,102,
102,102,103,119,119,136,153,153,154,170,153,152,135,119,119,119,119,119,137,153,
153,153,152,137,153,136,136,136,135,119,119,119,119,136,136,135,119,119,119,119,
119,119,119,119,137,152,136,136,153,154,169,153,136,136,136,135,119,102,85,85,
85,85,84,68,69,85,84,68,85,85,102,119,119,119,120,136,136,154,171,188,
204,205,221,222,237,221,204,187,170,153,136,119,119,119,102,101,85,85,68,68,
68,51,51,52,68,69,85,86,102,119,120,136,136,120,136,136,119,119,120,136,
136,136,136,136,136,136,119,119,119,119,136,136,136,136,136,136,136,136,136,136,
136,136,136,119,120
, // 3, click
136,136,136,136,136,136,136,136,136,136,136,136,136,135,119,136,136,119,119,
119,119,119,119,119,119,119,119,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,135,119,119,120,119,119,119,119,119,119,119,120,136,136,136,136,136,
136,136,136,136,136,136,135,136,119,136,136,119,119,120,135,119,119,119,120,119,
119,136,136,119,119,136,135,119,119,119,119,119,119,119,119,119,119,119,119,119,
119,119,119,136,136,119,120,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,135,119,136,136,135,119,119,120,136,135,119,120,119,119,120,135,120,119,
136,134,103,136,119,103,137,135,103,136,136,119,102,120,135,136,135,119,137,135,
119,102,154,133,67,54,154,136,150,69,120,120,133,72,169,119,118,86,171,132,
70,155,167,85,120,152,135,119,119,137,118,103,136,119,137,118,103,137,135,104,
152,136,135,119,136,136,119,120,152,120,119,152,152,120,120,136,135,120,135,119,
136,136,136,119,136,136,136,136,136,119,136,136,136,120,136,119,119,119,120,119,
119,119,119,119,119,119,119,119,119,119,119,136,135,119,135,119,136,120,136,136,
120,135,119,136,136,119,119,136,119,136,136,136,136,136,136,136,119,119,119,119,
119,119,119,119,119,119,119,136,136,136,136,136,136,136,136,136,136,136,119,119,
119,119,119,119,119,119,119,135,135,135,135,135,135,135,150,122,74,106,120,134,
134,165,150,135,120,120,120,120,119,120,119,119,120,119,119,119,119,119,119,119,
119,119,119,119,135,136,120,120,135,136,136,136,136,136,136,136,136,135,119,119,
119,136,119,119,120,120,136,136,136,136,136,136,136,136,120,136,120,136,136,120,
119,136,119,120,119,119,119,119,119,119,119,119,119,119,119,119,119,135,135,135,
135,135,135,119,119,120,105,104,118,150,135,135,119,136,120,120,136,135,136,136,
120,120,136,136,120,136,135,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,135,136,136,136,120,120,135,135,136,136,120,120,135,135,135,135,136,136,120,
120,120,136,120,120,135,136,136,135,135,135,136,136,135,136,136,120,120,136,120,
136,119,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,120,136,136,136,136,136,136,136,136,136,136,136,136,136,119,119,136,135,
120,136,120,136,120,135,120,136,136,135,135,120,135,135,120,120,119,136,119,136,
120,120,135,120,136,136,135,136,135,136,135,136,135,136,136,136,136,136,136,136,
136,120,120,136,135,120,136,120,136,136,136,120,135,135,135,136,135,120,119,136,
119,120,136,135,119,136,136,136,136,136,136,120,136,119,136,136,136,136,135,120,
136,120,136,136,119,136,135,120,136,120,120,136,119,136,136,136,136,135,136,135,
136,136,119,120,136,135,136,120,136,136,135,120,136,119,136,135,136,136,120,136,
136,136,120,136,136,135,135,135,135,135,137,167,122,102,90,195,138,87,120,150,
136,136,87,153,88,121,133,104,150,135,151,134,136,105,104,121,135,118,151,136,
119,136,119,121,135,120,120,120,134,152,119,120,135,120,135,119,136,136,119,135,
135,120,136,120,136,120,136,135,135,135,136,120,136,135,136,135,136,136,136,136,
136,119,136,135,120,136,136,136,136,135,136,136,120,136,120,136,135,136,136,135,
136,136,120,136,120,135,135,136,136,119,120,136,120,135,119,136,136,119,136,135,
120,136,135,136,135,119,136,135,136,136,135,120,136,120,136,135,136,120,136,135,
120,136,135,136,136,136,136,136,120,136,136,120,135,136,136,120,136,120,136,136,
136,136,136,136,136,120,136,136,136,136,136,136,136,136,136,136,136,135,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,120,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136
, // 4, plasma shot, teleport
136,136,136,136,136,136,119,119,118,102,102,85,86,102,103,120,136,153,170,
187,187,187,170,169,152,119,102,85,68,68,68,69,86,103,136,154,171,187,204,
203,186,169,136,118,102,85,84,69,103,138,189,221,221,221,221,221,221,221,221,
221,220,166,50,34,34,34,33,33,34,34,34,34,34,34,34,34,71,156,221,
221,221,221,221,221,221,221,221,221,222,238,238,238,221,237,184,99,51,34,34,
34,34,34,34,34,34,34,17,17,36,121,188,204,204,204,204,204,204,221,221,
221,221,221,221,221,221,221,238,221,219,169,136,101,67,51,51,51,51,50,34,
34,35,50,34,36,121,189,221,221,221,221,220,204,204,188,204,204,204,204,204,
204,204,205,220,186,152,117,50,51,51,51,51,51,51,51,51,51,51,52,105,
189,237,221,221,221,221,221,221,221,205,220,204,204,204,187,187,187,187,186,135,
101,50,17,17,18,34,35,51,67,51,51,51,52,105,187,222,221,221,221,221,
221,221,221,221,221,221,221,221,221,221,221,221,204,203,169,134,67,34,17,17,
17,34,34,33,34,34,34,52,104,171,205,221,221,220,204,204,204,204,204,204,
204,204,204,221,221,221,221,221,219,169,135,85,84,68,68,68,67,51,51,50,
34,35,69,103,136,136,136,136,135,136,136,136,153,170,187,187,187,187,187,187,
187,187,203,169,135,102,85,86,102,103,119,119,118,102,86,102,102,85,85,85,
84,68,69,85,102,103,137,154,170,171,186,170,170,170,170,170,152,118,101,68,
68,68,85,85,102,103,120,136,135,118,102,102,102,102,102,102,101,84,69,87,
137,154,153,170,170,171,187,187,187,187,170,170,169,170,170,170,170,153,152,135,
118,101,68,51,51,52,68,85,85,85,85,68,85,102,103,118,102,85,102,103,
120,136,153,170,187,204,221,238,238,238,237,221,204,186,152,118,101,84,68,69,
86,102,119,119,102,85,84,68,68,51,51,34,34,51,68,85,102,103,120,137,
154,171,188,205,221,204,204,203,187,186,170,153,153,153,153,153,136,136,119,119,
118,102,102,103,119,119,119,119,118,101,84,67,51,51,68,86,120,137,153,170,
170,153,152,135,119,136,136,119,119,119,119,119,119,119,119,136,137,153,153,136,
136,136,153,153,136,119,102,102,102,120,137,154,170,170,170,153,152,136,135,119,
119,102,102,103,119,102,85,84,69,85,102,119,119,119,119,119,119,119,119,102,
85,85,86,103,120,136,153,153,154,170,170,170,170,170,170,170,170,153,153,136,
135,119,119,120,136,136,136,136,136,136,136,119,102,101,85,85,85,85,102,120,
136,153,170,170,169,152,136,136,135,119,119,119,119,119,102,102,102,119,136,137,
153,153,153,153,136,136,119,118,102,102,102,102,119,119,136,137,154,170,169,136,
119,119,119,102,102,103,119,119,119,118,102,119,120,137,153,154,170,169,153,136,
136,136,135,119,119,119,119,119,102,119,120,136,119,118,102,103,119,119,119,136,
136,136,135,119,119,119,136,137,153,154,170,153,136,136,135,119,119,119,120,119,
119,119,103,119,119,135,119,119,119,118,102,102,102,119,119,119,119,119,119,119,
136,137,154,171,187,170,153,136,136,136,136,136,136,136,135,119,119,119,119,119,
119,119,119,118,102,102,102,119,119,119,119,119,119,119,120,136,153,153,170,169,
153,152,136,136,136,136,153,152,136,136,136,136,136,119,119,119,118,102,102,102,
102,102,103,119,119,118,102,102,119,120,136,153,154,170,169,153,153,153,153,153,
153,152,136,136,119,119,119,118,102,102,102,119,119,119,119,119,119,119,119,119,
119,119,119,119,119,136,137,153,153,153,153,153,153,153,153,136,136,136,136,136,
136,135,119,119,119,119,119,119,119,119,118,102,102,102,102,102,103,119,120,136,
136,136,136,153,153,136,136,136,137,153,153,153,152,136,136,136,136,136,135,119,
119,102,102,102,102,102,102,102,102,102,103,119,120,136,136,136,153,153,153,153,
136,137,153,153,136,136,136,136,136,136,136,136,119,119,119,118,102,102,102,102,
103,119,119,119,119,119,119,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,119,119,119,119,136,135,119,119,119,119,119,119,119,119,119,119,119,119,
119,120,136,136,136,136,136,136,136,136,136,136,136,136,136,135,119,119,119,119,
119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,119,120,136,136,136,
136,136,136,136,136,136,136,136,136,136,135,119,119,119,119,119,119,119,119,119,
119,119,119,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,136,
136,136,136,136,136
, // 5, robot/monster sound (beep)
136,120,120,120,136,136,136,136,120,120,135,136,136,136,120,136,136,119,136,
136,120,136,119,120,136,136,136,120,136,119,120,136,135,119,136,135,120,136,119,
119,136,136,119,119,120,136,136,135,119,119,136,136,136,119,119,119,136,119,120,
136,136,135,119,120,136,136,119,103,136,136,120,135,119,120,136,118,119,136,136,
136,119,119,136,136,119,120,135,103,136,136,119,136,135,119,120,136,119,120,135,
119,120,136,135,119,119,135,120,136,135,119,136,136,119,136,119,120,136,135,119,
119,119,136,136,119,120,119,136,136,119,119,119,119,137,136,120,135,102,120,136,
136,119,119,120,135,120,136,119,120,136,119,136,136,135,119,119,119,136,136,134,
104,136,136,119,119,119,136,135,120,136,120,119,119,119,135,136,136,136,119,119,
119,136,136,135,119,136,135,119,120,136,135,119,120,135,136,136,119,136,119,120,
136,135,119,120,135,119,136,135,119,136,136,136,120,119,119,136,119,120,136,135,
120,135,102,121,169,118,87,137,151,85,121,186,118,102,119,136,152,118,120,136,
135,119,120,120,136,136,136,136,101,103,153,152,135,102,120,136,135,102,120,153,
135,119,119,135,119,119,136,135,119,119,119,120,136,135,119,119,136,136,136,119,
119,119,136,119,119,135,120,135,119,119,120,135,120,136,119,119,136,136,135,119,
135,119,136,136,136,136,135,119,120,136,119,120,136,118,119,137,135,119,120,135,
119,136,135,119,120,152,119,136,119,120,135,136,119,136,119,136,135,120,152,119,
120,136,119,119,154,150,103,152,119,120,153,135,102,138,135,136,119,153,101,121,
151,120,135,136,118,137,151,103,120,152,102,138,134,88,154,151,103,153,118,104,
137,135,120,119,153,117,105,167,102,120,136,119,137,134,87,153,135,120,152,102,
103,137,151,119,153,133,87,169,100,104,153,152,102,103,153,135,119,136,119,120,
136,119,119,135,119,120,136,119,119,119,120,136,135,122,222,219,132,51,68,87,
171,187,186,134,50,52,104,172,203,186,116,33,54,139,188,204,151,67,52,87,
155,220,152,136,98,17,73,223,218,136,100,34,53,155,187,188,185,82,18,71,
155,205,203,150,65,19,86,156,238,201,100,50,36,122,205,220,167,66,34,88,
171,204,201,102,84,51,71,155,221,184,118,66,19,107,238,202,152,99,1,72,
172,204,218,116,51,36,104,172,222,183,83,35,87,120,190,234,118,100,34,70,
155,237,170,167,50,52,104,155,221,168,100,51,69,103,190,236,134,82,2,120,
138,223,199,102,82,19,105,204,204,185,82,52,85,106,216,103,153,117,104,134,
104,152,118,103,137,135,136,136,135,102,119,121,151,104,152,84,104,135,154,134,
103,136,119,135,120,135,120,135,136,119,136,118,86,153,118,137,134,137,117,105,
153,136,117,103,136,120,152,119,119,136,135,136,135,104,152,137,136,119,119,120,
119,135,86,156,204,253,150,67,35,87,155,222,200,84,52,51,89,206,236,186,
115,34,69,104,206,220,150,84,67,53,141,254,168,118,50,53,104,188,186,151,
84,68,87,172,220,169,135,66,53,120,154,188,203,116,52,84,71,189,219,169,
116,34,53,122,207,236,167,66,35,86,155,221,184,101,66,53,137,172,204,169,
116,35,69,105,190,219,150,83,51,71,172,204,203,149,49,35,104,172,238,200,
100,51,69,105,205,218,134,83,52,88,172,204,185,117,84,51,104,172,204,185,
100,51,52,106,221,203,152,101,50,54,139,204,186,150,67,68,86,156,221,202,
117,34,52,122,200,135,119,120,136,118,103,137,152,135,119,119,136,152,118,103,
136,135,119,136,135,120,135,118,120,135,103,154,151,102,102,119,136,154,169,118,
102,119,119,120,152,118,119,136,154,151,101,121,152,102,136,120,136,119,119,119,
120,136,118,103,137,152,119,119,136,136,119,119,118,119,120,136,153,135,103,136,
119,136,136,119,119,120,136,136,135,136,119,119,120,136,119,136,135,120,135,119,
120,136,135,136,119,119,136,136,119,136,135,119,136,119,120,136,135,119,136,119,
119,136,135,119,118,120,153,153,151,118,119,119,119,136,135,119,136,136,136,135,
120,136,135,119,119,119,136,136,119,120,135,119,136,136,119,119,119,119,137,152,
135,119,119,119,136,136,119,119,119,136,136,136,119,120,136,119,119,119,136,136,
119,119,119,136,136,136,135,119,136,119,120,136,119,136,135,120,119,136,135,120,
135,119,136,119,136,135,119,136,119,119,120,136,136,119,120,135,136,135,120,135,
119,136,119,136,135,136,119,120,136,120,136,120,136,119,135,119,136,119,136,135,
136,119,120,135,120
};

#endif // guard
