/**
  @file assets.h

  This file containts assets to be used in the game. Textures are stored by
  columns for cache friendliness (as rendering also happens by columns),
  4 bits per pixel, which means an index to 16 color subpalette stored at the
  beginning of the image. Images can be converted to this format with the
  provided pything script:

  python img2array.py -t -c -x32 -y32 -ppalette565.png image.png

  by Miloslav Ciz (drummyfish), 2019

  Released under CC0 1.0 (https://creativecommons.org/publicdomain/zero/1.0/)
  plus a waiver of all other intellectual property. The goal of this work is to
  be and remain completely in the public domain forever, available for any use
  whatsoever.
*/

#ifndef _SFG_IMAGES_H
#define _SFG_IMAGES_H

#define SFG_TEXTURE_SIZE 32

#define SFG_TEXTURE_STORE_SIZE (16 + (SFG_TEXTURE_SIZE * SFG_TEXTURE_SIZE) / 2)

/**
  Color index which will in textures and sprites be considered transparent.
*/
#define SFG_TRANSPARENT_COLOR 175

/**
  Special index of an implicit texture that consists of only transparent pixels.
*/
#define SFG_TRANSPARENT_TEXTURE 255

static inline uint8_t SFG_getTexel(const uint8_t *texture, uint8_t x, uint8_t y)
{
  x &= 0x1f;
  y &= 0x1f;

  return SFG_PROGRAM_MEMORY_U8(texture +
    ((SFG_PROGRAM_MEMORY_U8(texture + 16 + (x * SFG_TEXTURE_SIZE + y) / 2) >> 
     (4 * (y % 2 == 0))) & 0x0f));
}

#define SFG_WALL_TEXTURE_COUNT 16

SFG_PROGRAM_MEMORY uint8_t
  SFG_wallTextures[SFG_WALL_TEXTURE_COUNT * SFG_TEXTURE_STORE_SIZE] =
{
// 0, white wooden planks
4,21,5,3,20,0,2,26,6,18,57,65,1,16,17,49,0,1,0,3,32,1,33,0,161,64,1,16,64,3,3,
32,0,17,16,9,32,1,17,1,161,65,1,16,78,132,35,32,0,17,16,7,32,1,32,17,240,49,0,
16,57,99,87,206,0,18,16,6,33,0,16,1,177,48,0,0,97,0,2,0,0,17,32,6,32,1,0,1,178,
48,0,0,48,0,2,0,0,17,32,7,32,17,1,0,178,48,1,0,49,0,1,0,0,1,16,3,32,18,0,0,194,
48,0,0,48,0,18,4,64,17,32,4,35,0,3,0,194,48,0,0,49,0,34,4,0,17,32,6,130,50,32,
40,242,48,1,0,49,0,18,0,0,2,32,12,85,85,85,85,81,48,2,0,49,0,33,3,0,2,16,3,32,0,
32,2,98,48,1,0,49,0,34,19,0,18,17,3,32,0,32,0,194,48,2,0,0,0,34,3,4,2,16,3,16,0,
32,32,226,48,2,0,48,0,33,19,0,18,33,3,0,16,32,32,178,48,1,1,50,1,34,3,4,17,16,3,
0,16,32,16,162,48,1,64,50,1,34,7,3,1,17,3,0,16,32,16,184,2,2,2,2,48,18,3,0,0,17,
3,0,17,32,16,94,93,93,89,112,0,18,3,0,0,16,3,0,2,32,16,48,48,48,3,0,0,17,4,0,0,
17,3,0,1,32,16,48,48,0,3,1,1,18,3,0,0,32,7,16,1,32,16,48,0,1,0,1,50,34,3,0,1,33,
3,16,1,20,32,112,48,0,16,1,49,34,3,64,16,33,3,4,2,4,16,48,49,0,16,0,49,34,3,64,
0,32,3,16,1,0,0,48,49,0,16,65,48,34,3,49,0,17,3,16,2,0,0,48,49,1,32,0,49,34,3,
64,0,16,19,16,2,1,0,113,49,17,16,0,48,32,3,0,0,16,19,16,1,2,0,97,66,17,32,64,48,
33,19,34,2,0,137,16,1,18,0,97,49,17,16,66,64,32,3,84,212,105,69,16,1,17,0,97,49,
18,16,50,48,16,19,51,48,51,51,16,1,16,0,98,50,1,19,66,48,32,3,0,1,0,3,32,1,33,0,
177,49,2,16,66,64,16,3,0,1,0,7,32,1,16,1,82,64,1,16,50,0,16,35,0,1,0,9,33,2,32,
0,161,64,1,20,48,0,0,32
, // 1, skyscraper window
4,5,3,6,2,81,0,59,48,128,138,131,66,44,137,129,16,176,208,176,44,33,0,0,1,43,1,
13,0,32,29,0,0,176,0,0,12,64,0,17,0,32,0,17,1,32,13,11,0,176,0,208,4,65,0,0,0,
34,1,16,1,32,16,0,23,32,0,2,40,66,2,194,230,140,34,208,2,176,17,0,119,76,66,34,
34,34,34,34,34,34,34,34,34,116,64,0,72,194,51,51,51,51,17,49,51,19,19,49,51,12,
34,36,17,210,49,17,16,17,1,17,17,16,17,1,17,42,1,17,17,2,49,154,116,244,229,164,
233,85,49,16,17,14,1,16,2,69,17,104,79,117,149,88,94,137,49,17,17,36,0,0,196,69,
48,137,119,69,84,79,88,85,49,17,1,4,92,244,0,34,0,69,85,95,127,116,86,89,49,17,
17,36,0,0,17,32,1,169,88,229,228,74,85,85,49,19,1,4,0,0,0,2,49,90,153,85,85,169,
95,85,49,16,17,4,2,0,0,39,48,68,85,233,138,37,133,136,49,17,17,36,0,32,0,4,17,
122,150,84,133,230,136,136,49,1,17,10,0,0,0,44,48,35,51,51,51,51,51,35,49,17,1,
66,0,13,0,4,17,51,3,211,3,3,35,2,49,17,17,34,0,0,0,37,49,167,42,70,106,69,102,
102,49,0,17,36,2,34,68,244,48,39,119,79,69,150,102,102,49,17,17,68,64,34,16,76,
49,119,2,127,120,102,102,102,49,17,19,5,0,1,1,74,49,114,119,4,152,102,104,102,
49,17,17,75,0,0,208,28,49,119,39,116,248,104,134,102,49,16,17,34,0,0,0,10,17,39,
114,37,169,104,136,86,49,16,17,4,0,0,1,43,49,114,126,232,74,102,104,134,49,1,0,
34,0,32,0,116,17,119,122,102,88,102,102,102,49,16,17,66,0,0,1,36,49,17,17,17,17,
17,17,17,1,1,1,64,0,0,32,184,49,17,49,17,17,19,51,17,17,17,19,178,0,34,44,66,34,
71,39,34,180,36,66,34,2,32,32,192,17,19,66,192,0,16,176,66,34,34,34,200,36,34,
34,36,68,44,221,192,0,0,10,32,0,16,1,34,16,16,0,11,0,16,13,64,0,0,10,0,0,0,1,34,
0,1,1,2,16,0,1,176,27,0,11,33,0,17,0,112,0,1,17,113,16,0
, // 2, pink high-tech wall
83,4,12,5,84,20,61,85,13,6,11,3,7,14,15,0,34,32,34,0,32,2,170,34,170,160,170,
160,17,17,17,17,34,34,32,34,42,34,170,2,42,42,170,170,17,17,17,17,34,34,34,36,
34,160,36,32,160,160,42,10,49,17,17,17,34,34,34,42,34,0,0,0,0,0,0,4,49,49,17,17,
34,34,2,10,36,32,17,17,81,17,81,23,59,59,59,17,0,0,0,0,2,0,84,84,96,0,0,0,59,59,
49,19,102,102,102,102,96,32,20,86,0,102,102,102,145,59,59,19,34,34,34,36,96,32,
21,86,5,85,85,23,59,59,49,19,34,34,34,42,96,64,21,86,5,85,69,71,145,155,49,49,
34,34,66,37,96,128,84,86,5,85,91,23,49,49,49,19,34,34,0,0,0,0,0,0,4,84,69,87,
145,155,49,49,34,38,6,102,102,102,102,96,68,84,84,23,145,145,145,51,34,38,2,37,
88,32,68,69,69,85,85,71,49,145,49,51,40,134,2,34,130,64,21,69,85,84,21,23,145,
145,147,51,36,38,4,68,68,64,84,85,85,84,20,71,145,145,49,51,37,38,2,66,68,32,69,
85,85,85,69,87,145,145,147,51,40,134,5,84,72,64,20,65,69,68,20,71,145,193,147,
51,36,70,2,72,136,112,68,65,20,84,20,87,145,145,51,57,40,134,8,133,132,32,84,84,
17,84,68,87,195,147,147,51,40,134,8,88,136,112,21,17,17,68,17,23,147,51,51,51,
36,134,4,133,136,112,81,17,17,68,65,87,147,147,51,57,40,134,0,0,0,0,0,0,65,20,
68,71,147,57,147,57,39,120,102,102,102,102,102,102,1,17,84,71,153,51,57,57,40,
119,135,119,96,112,17,22,1,17,68,183,147,57,51,51,37,120,135,120,96,112,177,22,
1,17,17,87,153,147,57,57,37,136,136,135,96,32,65,22,1,20,91,71,60,57,57,57,0,0,
0,0,96,112,84,22,96,0,0,0,57,57,60,57,102,102,102,102,104,128,75,181,102,102,
102,102,60,57,51,60,39,119,135,119,120,112,180,84,180,181,75,183,57,60,57,57,34,
136,135,215,231,116,119,119,119,119,119,119,60,51,60,60,37,120,135,114,119,135,
125,120,116,212,68,114,60,60,57,57,34,117,136,135,135,120,132,136,136,136,136,
71,57,57,57,153
, // 3, partly mossy concrete wall 
4,3,5,36,20,37,76,12,11,19,77,130,13,35,2,0,2,0,0,80,32,0,0,1,5,37,1,64,4,25,19,
145,0,32,96,32,0,16,48,24,115,48,65,53,48,16,113,11,2,0,0,0,0,0,0,152,3,48,49,
48,51,19,1,17,0,32,16,48,0,16,0,8,0,0,1,67,0,23,19,17,2,0,5,48,0,16,0,24,115,65,
113,85,67,16,0,17,2,0,18,0,0,0,0,6,4,0,0,67,48,19,145,17,0,0,3,80,80,0,0,1,3,80,
19,3,83,16,9,1,2,0,210,0,0,16,0,1,148,3,1,48,51,16,17,17,19,0,16,48,0,16,0,24,5,
64,113,51,3,49,1,11,2,0,0,34,0,0,0,8,85,3,1,67,84,3,112,17,0,36,5,4,32,96,0,8,4,
4,3,52,83,16,17,1,34,0,2,32,0,0,7,1,3,32,49,3,51,119,49,113,4,32,96,66,0,0,0,7,
5,84,1,51,67,112,0,30,2,32,98,32,10,0,0,8,68,32,64,5,3,49,0,14,2,2,96,0,0,96,0,
6,5,80,49,48,51,112,1,27,0,32,2,0,1,96,0,1,51,0,1,3,51,27,0,27,2,0,96,32,0,0,6,
8,0,48,3,3,48,208,24,1,2,32,98,4,32,0,0,6,4,36,3,0,51,0,16,17,34,194,96,34,0,96,
0,6,50,48,1,83,67,144,0,8,0,42,2,36,0,0,162,8,5,32,65,64,48,23,1,1,2,32,100,34,
6,0,12,8,68,50,1,0,3,16,112,1,36,34,2,0,0,0,0,8,0,0,0,48,0,23,7,17,2,42,4,32,42,
0,0,8,5,64,64,35,0,135,0,1,2,64,98,44,6,0,32,8,48,48,1,85,0,17,0,49,2,32,3,2,0,
4,0,8,5,64,0,0,4,0,112,1,4,34,98,2,6,98,32,8,85,36,48,67,5,1,0,11,2,32,98,192,
160,4,52,8,4,0,1,4,0,0,112,113,2,194,2,0,0,0,2,1,5,85,64,211,4,48,0,1,2,32,0,42,
0,21,4,8,64,68,45,4,48,0,23,27,2,32,96,0,0,16,0,1,5,80,0,85,3,16,0,1,2,194,0,0,
0,16,48,17,84,85,0,48,64,144,25,17,0,32,96,0,0,0,64,13,3,68,0,5,48,16,0,17
, // 4, wooden chess pattern 
20,12,11,21,2,43,19,73,1,83,81,10,9,34,42,65,86,82,91,85,89,249,149,153,66,34,
40,34,34,114,34,36,0,0,0,48,48,35,0,2,33,18,39,17,17,66,17,18,0,0,5,48,3,96,3,
50,33,34,40,33,33,113,17,18,96,0,6,48,3,99,3,50,33,34,23,33,18,129,17,18,80,3,5,
48,3,99,3,54,33,33,23,18,17,114,17,18,80,0,5,48,3,99,0,54,33,34,20,34,33,114,33,
18,80,0,2,48,51,163,3,54,65,34,20,34,33,65,33,18,80,3,2,48,0,163,0,54,33,34,20,
34,17,66,33,18,0,0,2,3,0,67,0,53,177,34,20,34,17,130,33,18,0,0,2,51,0,211,0,5,
33,34,36,34,17,114,18,18,80,0,6,3,0,99,0,53,33,34,20,18,33,113,17,18,0,3,2,48,3,
99,0,6,65,33,20,18,33,65,17,18,0,0,2,0,0,96,0,6,34,34,20,17,33,129,17,18,0,0,5,
0,0,99,0,1,33,34,36,17,33,130,17,18,3,3,53,48,0,35,0,6,65,17,20,17,17,66,17,18,
80,85,86,5,85,224,85,85,68,68,72,66,68,132,68,36,66,34,20,34,34,66,34,36,96,0,5,
5,80,80,85,82,65,34,20,33,34,65,18,18,3,3,53,48,3,83,0,5,65,34,20,18,34,65,18,
34,51,51,54,51,3,80,5,5,66,34,20,18,34,65,18,18,51,51,53,51,3,80,5,5,66,34,20,
17,33,65,17,18,0,51,54,51,3,80,5,5,65,34,20,17,17,65,17,34,51,51,53,51,3,80,0,0,
65,34,20,17,17,65,17,18,0,51,54,48,51,80,0,0,66,34,20,17,18,65,17,18,3,51,54,48,
51,80,5,5,65,34,20,17,33,65,17,18,3,51,54,51,51,32,0,5,65,34,20,17,33,65,17,18,
3,51,54,51,51,80,1,5,65,34,20,17,33,65,34,34,3,51,59,48,51,80,5,85,65,34,20,17,
17,66,34,36,3,51,52,48,3,83,1,5,65,34,20,17,17,68,17,20,3,51,54,48,48,35,48,5,
65,33,20,34,17,66,17,18,3,0,52,51,51,35,48,5,65,33,20,17,17,65,18,18,3,51,54,51,
0,35,48,5,32,80,106,0,80,101,6,2,85,85,92,86,86,160,5,82
, // 5, red brick wall
13,5,6,21,93,101,4,100,11,7,19,3,106,178,0,0,33,17,22,37,48,8,33,3,54,33,64,0,1,
20,5,70,37,0,88,37,85,88,32,64,4,17,64,0,65,20,0,70,32,51,56,37,85,8,33,0,4,17,
83,51,81,20,96,70,32,3,52,38,51,8,33,3,4,18,83,51,81,20,85,70,38,6,4,32,51,8,33,
3,4,18,64,51,81,20,85,69,38,0,4,38,3,52,33,83,5,17,67,51,81,20,101,70,38,0,4,38,
83,4,17,83,0,17,67,51,81,20,101,68,38,0,4,38,69,68,17,86,80,18,69,85,65,36,101,
68,32,85,4,34,17,17,33,86,83,18,18,34,34,36,0,88,33,0,4,34,34,18,33,83,0,18,17,
17,17,36,0,84,35,3,56,34,64,76,17,83,48,98,112,0,119,36,0,68,35,0,4,34,133,4,18,
3,48,98,113,0,17,36,0,85,33,0,4,34,133,100,17,81,48,98,112,0,1,36,0,86,35,0,8,
33,165,68,17,67,53,98,113,0,1,36,85,86,35,0,4,33,69,106,17,64,53,98,112,0,1,20,
85,86,35,0,4,33,133,68,17,64,53,18,112,0,1,20,5,86,35,0,4,33,166,84,17,64,53,18,
112,0,1,20,5,86,35,3,4,33,69,10,17,80,53,18,112,0,1,36,85,86,35,0,52,34,69,100,
33,83,5,18,112,0,113,40,5,84,33,64,8,33,69,4,33,69,68,18,112,0,1,36,0,22,33,0,4,
33,0,52,17,34,34,18,112,0,1,34,34,145,34,41,146,33,0,4,17,17,17,18,113,17,1,36,
99,22,38,80,49,33,69,4,32,0,4,98,112,17,17,36,51,6,32,80,4,33,69,106,35,3,53,98,
112,0,17,36,51,6,33,3,52,33,69,72,35,51,53,18,16,0,17,37,51,6,33,3,52,17,69,84,
35,51,53,98,112,0,17,37,51,6,33,1,52,17,69,100,35,51,48,98,113,17,17,21,3,6,35,
3,52,17,67,4,35,51,48,98,112,0,17,16,51,6,35,3,53,17,69,100,33,49,19,97,112,0,
17,37,51,6,35,3,4,17,212,4,33,3,51,98,1,0,1,36,3,6,35,3,52,18,34,34,33,3,48,177,
34,153,146,36,51,54,33,3,52,34,49,17,33,3,48,98,70,102,17,36,0,6
, // 6, grass-covered concrete wall, tiles with skyscraper window 
36,37,43,35,4,3,107,34,131,50,42,5,106,110,114,26,3,131,51,128,58,52,0,5,84,56,
84,2,0,48,66,0,48,128,5,5,10,112,0,68,0,48,85,68,4,144,32,1,6,32,17,0,7,116,80,
0,85,153,36,13,0,1,0,0,16,1,1,0,54,115,3,160,110,234,55,1,208,40,1,0,170,96,0,
39,115,51,51,144,0,51,144,0,0,16,0,2,17,16,17,208,11,187,20,0,2,43,64,32,17,16,
0,44,17,1,0,16,3,1,0,2,0,3,64,0,0,32,32,0,1,16,17,0,1,3,2,96,32,37,64,0,2,0,2,
32,17,16,1,1,0,0,34,128,32,0,1,16,16,32,0,32,1,0,0,0,16,8,0,2,2,2,2,0,32,2,32,0,
34,1,19,48,5,96,0,2,0,32,0,17,1,2,0,16,1,18,16,65,16,0,32,0,1,2,2,4,0,8,32,2,1,
39,18,4,1,32,0,208,0,0,0,1,16,0,2,0,0,18,19,48,0,16,13,0,1,0,32,18,16,0,0,0,16,
1,0,0,16,65,16,0,0,33,1,16,0,54,0,38,0,6,64,0,129,0,0,0,0,1,0,0,0,9,63,50,2,2,8,
2,32,0,18,0,17,0,1,17,96,0,5,80,32,0,0,0,0,18,69,80,0,1,16,48,1,0,32,153,206,32,
2,2,32,17,4,64,96,32,0,17,0,224,0,131,64,0,0,0,1,0,4,8,0,0,16,16,1,18,96,0,2,2,
0,34,32,2,0,0,5,0,0,96,2,0,32,0,2,0,2,0,17,0,0,0,85,34,32,0,0,0,0,0,2,0,1,16,48,
0,33,2,68,0,2,2,0,16,0,32,32,32,38,1,32,0,0,34,6,0,0,48,3,0,8,0,0,0,18,16,0,0,
98,0,32,0,0,0,33,32,0,32,0,17,17,0,0,0,33,32,0,0,4,49,18,0,2,16,2,1,66,0,98,2,0,
0,0,34,0,17,1,0,0,0,17,0,17,33,2,0,12,126,199,112,39,0,16,6,21,0,16,16,0,16,32,
32,167,51,63,62,172,151,113,0,0,2,0,1,17,48,16,16,48,69,4,51,64,64,0,16,8,96,27,
16,16,16,62,57,149,0,4,51,0,1,4,7,80,2,0,17,1,51,8,52,0,68,0,32,0,4,68,36,16,32
, // 7, steel door
4,80,3,17,5,59,6,69,58,50,60,2,68,74,70,67,1,17,17,17,17,17,17,17,17,17,17,17,
17,17,17,17,22,102,70,102,70,102,70,102,70,100,102,100,102,100,102,97,16,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,1,16,0,32,0,32,0,32,0,32,2,0,2,0,2,0,1,19,189,187,221,189,
219,219,219,187,187,187,219,189,189,189,177,17,51,51,51,51,51,51,51,51,51,51,51,
51,51,51,49,19,136,136,136,133,133,85,95,34,34,44,47,34,194,34,40,19,136,34,34,
34,2,34,34,34,32,34,32,2,2,2,34,19,130,40,140,136,192,197,192,112,112,64,0,0,7,
4,7,19,146,136,200,92,80,122,90,160,0,64,0,7,0,4,0,19,146,149,136,200,84,122,
202,0,0,64,0,0,7,14,0,19,146,153,153,197,94,170,80,112,0,64,0,112,0,4,0,19,146,
37,156,133,112,5,0,0,224,224,0,112,112,4,0,19,146,153,153,87,160,0,7,160,0,64,0,
0,0,4,4,19,146,149,41,149,4,7,80,160,10,71,0,0,7,4,0,19,146,153,146,87,0,170,87,
0,7,64,0,7,0,4,4,19,146,89,204,87,4,167,80,7,0,224,0,0,0,4,0,19,146,153,156,90,
160,122,80,0,14,64,0,0,112,116,4,19,146,149,153,85,116,170,192,0,112,64,112,0,0,
4,0,19,146,41,92,138,84,122,90,0,0,71,0,7,0,4,4,19,146,153,204,80,126,90,202,
224,0,64,0,0,0,4,0,19,146,143,85,32,4,170,90,160,0,71,0,7,7,14,0,19,130,136,140,
128,116,122,202,112,10,64,0,112,0,4,0,19,130,40,200,92,14,170,80,0,170,74,112,0,
112,116,0,19,136,34,34,34,2,2,0,0,32,34,34,34,34,2,34,19,136,136,136,133,34,85,
85,85,37,194,47,34,242,34,34,17,51,51,51,51,51,51,51,51,51,51,51,51,51,51,49,27,
189,221,187,219,219,219,187,221,187,187,221,189,189,189,177,22,102,70,102,70,
102,70,102,70,102,70,100,102,100,102,97,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,16,0,
32,0,32,0,32,0,32,0,32,2,0,2,0,1,1,17,17,17,17,17,17,17,17,17,17,17,17,17,17,17
, // 8, white skull on blue
6,5,4,59,60,62,218,14,3,13,85,139,74,63,12,15,17,17,17,17,17,18,147,69,72,17,17,
17,17,17,17,17,16,16,1,1,16,18,35,51,50,17,1,1,16,16,1,17,16,0,0,0,0,18,36,99,
82,16,16,0,0,0,0,17,0,0,0,0,0,18,37,51,82,17,0,0,0,0,0,1,16,0,0,0,0,18,35,179,
50,17,16,0,0,0,0,1,0,0,0,0,0,18,35,51,66,17,16,0,0,0,0,17,16,0,0,0,1,226,37,67,
84,40,17,0,0,0,0,1,0,0,0,17,42,34,85,83,85,67,33,16,0,0,0,17,0,0,1,18,34,51,34,
34,68,75,98,129,16,0,0,17,16,0,17,162,51,130,34,34,38,198,52,66,129,0,1,17,17,
17,42,36,98,33,17,17,34,51,51,68,98,33,17,17,130,34,36,68,33,17,17,23,121,34,52,
68,51,66,34,34,68,84,53,85,33,0,0,121,151,34,37,85,67,69,221,51,51,68,52,82,16,
0,0,126,151,18,34,133,91,69,84,99,51,107,52,66,0,16,16,247,119,17,34,40,60,67,
51,68,83,99,51,50,1,1,17,0,1,33,18,40,51,51,52,85,75,52,52,66,0,17,17,0,1,33,18,
40,51,68,51,69,51,84,53,82,1,0,16,247,119,17,34,40,70,69,51,51,85,84,52,82,16,0,
0,126,151,18,34,132,70,69,84,99,85,68,52,52,33,0,0,121,151,34,36,52,60,52,85,51,
130,34,35,107,33,17,17,23,121,34,99,107,195,66,34,34,17,17,42,102,98,33,17,17,
34,54,102,102,98,33,17,17,16,0,17,162,68,130,34,34,36,76,102,98,33,0,1,17,0,0,1,
18,35,69,34,34,85,67,194,161,16,0,0,17,16,0,0,17,42,35,69,67,68,75,33,16,0,0,0,
17,0,0,0,0,1,226,35,99,54,42,17,0,0,0,0,1,0,0,0,0,0,18,35,51,66,17,16,0,0,0,0,
17,16,0,0,0,0,18,37,51,82,17,16,0,0,0,0,1,0,0,0,0,0,18,45,51,66,17,0,0,0,0,0,1,
16,0,0,0,0,18,36,99,50,16,16,0,0,0,0,17,16,16,1,1,16,18,35,68,50,17,16,16,16,16,
1,17,17,17,17,17,17,18,148,85,88,17,17,17,17,17,17,17
, // 9, red lava with stones
3,4,12,11,83,94,102,92,91,14,93,15,19,20,100,0,3,82,2,83,32,67,84,1,17,17,16,5,
32,17,20,0,2,48,0,67,117,118,36,0,33,33,0,5,33,17,17,19,69,64,17,20,67,86,82,36,
17,1,0,53,1,17,17,64,56,1,17,17,0,130,50,117,131,64,4,39,0,17,17,0,32,1,17,17,3,
82,4,68,39,85,34,98,36,16,0,3,0,0,1,16,3,82,1,16,0,35,53,106,83,51,51,54,85,68,
36,3,53,32,49,17,16,0,54,34,66,35,37,86,67,85,85,34,38,66,1,17,17,0,82,64,64,0,
3,101,0,32,3,34,102,103,65,1,17,3,84,1,17,0,3,83,18,17,0,67,34,68,85,52,0,69,48,
17,17,19,69,0,17,17,20,3,112,0,2,53,51,114,1,17,17,16,69,65,17,17,16,3,80,3,0,4,
46,98,64,17,17,32,115,1,17,18,0,53,64,17,17,16,50,86,34,0,0,0,35,1,0,0,0,37,65,
17,17,17,2,34,119,85,67,66,34,0,51,130,50,98,64,17,17,17,2,35,68,51,85,102,98,
35,51,50,86,102,35,34,2,16,3,132,0,0,4,34,82,35,20,0,66,85,39,120,34,4,2,80,1,
17,4,53,52,0,17,4,3,83,64,2,37,81,69,16,77,17,16,69,16,0,18,16,0,84,16,0,67,66,
38,49,17,17,32,53,32,17,17,17,0,82,0,33,16,3,102,32,0,0,4,84,0,18,17,17,0,84,1,
17,17,4,182,98,64,0,3,84,1,17,17,17,0,83,1,17,17,4,99,37,83,50,50,32,0,33,17,16,
0,83,1,17,33,2,36,0,34,114,118,84,16,17,0,0,3,83,1,17,16,67,49,1,16,4,38,103,51,
64,84,68,53,32,0,17,0,55,32,17,17,0,39,50,39,85,37,82,166,98,36,16,64,34,1,17,
17,16,55,32,0,36,0,4,54,34,117,66,51,84,1,17,17,16,82,0,0,2,17,4,39,64,18,82,
118,98,64,1,17,4,84,16,17,17,18,4,84,4,0,20,53,102,82,32,0,4,84,0,33,17,16,2,32,
1,17,64,39,51,69,85,117,53,156,65,1,18,50,38,36,17,17,16,34,0,0,20,67,102,101,
115,51,51,130,101,101,32,17,4,84,1,17,33,16,38,148,19,56,56
, // 10, transparent window
175,4,5,3,59,13,49,60,74,62,61,50,83,2,69,12,17,17,33,17,81,17,17,33,17,17,17,
17,18,34,34,17,59,17,17,17,17,17,17,17,17,17,177,17,18,34,82,33,51,51,51,51,51,
51,51,51,51,51,59,81,18,34,33,17,49,182,102,102,102,102,102,102,102,107,18,33,
18,34,34,33,51,0,0,0,0,116,16,0,0,10,50,37,18,34,34,33,51,0,0,0,0,116,16,0,0,10,
50,17,18,34,21,33,51,0,0,0,0,84,16,0,0,10,50,33,17,243,51,52,51,0,0,0,0,20,16,0,
0,10,50,145,18,17,17,21,51,0,0,0,0,20,32,0,0,10,50,149,18,49,81,17,51,0,0,0,0,
116,112,0,0,1,50,37,82,17,113,17,51,18,51,140,140,71,72,140,136,200,50,37,82,49,
17,225,51,18,68,68,68,121,116,68,68,71,50,33,82,18,18,17,51,0,0,0,0,212,208,0,0,
12,50,17,18,37,34,17,51,0,0,0,0,20,16,0,0,9,50,17,18,18,37,30,51,0,0,0,0,116,32,
0,0,2,50,145,18,34,34,33,51,0,0,0,0,116,16,0,0,2,50,37,18,34,18,17,51,0,0,0,0,
116,16,0,0,10,50,145,18,37,33,18,51,0,0,0,0,116,16,0,0,10,57,145,82,33,18,225,
51,0,0,0,0,20,16,0,0,10,57,145,18,18,17,21,51,0,0,0,0,212,208,0,0,13,50,149,18,
37,18,17,51,18,68,68,68,121,116,68,68,71,50,37,18,34,33,17,51,18,60,200,140,71,
76,136,136,200,50,37,18,34,81,30,51,0,0,0,0,244,16,0,0,1,50,149,18,34,33,17,51,
0,0,0,0,20,32,0,0,10,50,33,18,18,17,17,51,0,0,0,0,116,32,0,0,2,50,33,17,243,51,
52,51,0,0,0,0,20,16,0,0,10,50,17,18,17,17,17,51,0,0,0,0,84,16,0,0,10,50,33,82,
17,33,31,51,0,0,0,0,116,16,0,0,2,50,33,18,17,17,17,49,182,102,102,102,102,102,
102,102,107,34,81,18,37,17,17,19,50,34,34,34,34,34,34,34,34,35,81,18,34,17,17,
59,34,34,34,34,34,34,34,34,34,181,17,18,18,81,21,17,17,17,17,33,17,17,17,21,17,
17,17,18,34,34,18
, // 11, white steel blocks
6,5,7,4,3,85,71,78,0,0,0,0,0,0,0,0,68,68,68,68,68,68,68,68,68,68,68,68,68,68,68,
68,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51,52,34,0,34,34,2,34,34,34,34,34,
0,34,34,34,0,17,0,17,0,0,0,1,16,0,0,0,0,0,1,16,17,17,0,20,0,0,0,0,0,0,0,0,0,0,1,
64,0,17,0,0,0,1,0,0,0,0,0,1,0,0,0,0,0,17,17,17,17,17,17,17,17,81,17,17,17,17,17,
17,17,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,17,1,17,16,0,0,0,0,0,0,0,0,0,0,0,0,17,0,
0,1,1,0,0,6,0,0,0,0,0,0,0,0,17,34,0,34,0,34,34,34,34,34,34,34,34,32,2,34,17,0,
17,0,1,16,0,0,0,0,0,17,0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,17,0,0,0,0,1,0,
0,0,0,0,0,0,16,0,17,17,17,17,17,81,17,22,0,0,0,1,17,1,17,17,17,19,0,0,0,0,0,2,0,
0,0,1,0,0,0,0,113,21,16,0,0,0,0,2,0,0,0,1,0,0,0,0,0,19,0,0,0,0,0,2,0,0,0,1,0,0,
0,0,1,17,34,34,32,2,34,34,0,0,0,2,34,34,0,34,0,17,0,0,0,0,0,17,0,0,0,0,0,1,16,
16,17,17,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,17,0,1,0,0,0,0,0,0,0,0,0,0,0,17,17,
17,17,17,17,17,17,81,17,17,17,17,17,17,17,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,1,
17,16,0,0,0,0,0,0,0,0,1,0,1,16,17,0,0,1,1,0,0,6,0,0,0,16,0,0,0,0,17,34,0,34,0,
34,0,34,34,34,34,34,34,32,2,34,17,0,17,1,1,16,0,0,0,0,0,17,0,1,16,0,17,0,20,0,0,
0,0,0,0,0,0,0,0,1,64,1,17,0,0,0,0,0,0,0,1,0,0,0,0,16,0,17,17,17,17,17,17,17,17,
17,17,21,17,17,17,17,17,17,19,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51,51
, // 12, gray square-tiled wall
4,5,3,13,50,6,12,21,85,20,14,11,19,22,84,0,17,17,17,17,17,17,17,18,17,17,17,17,
17,17,17,20,17,3,16,8,17,16,0,18,17,16,1,1,0,3,16,18,16,1,16,1,0,49,17,48,17,1,
3,6,0,17,3,2,16,16,0,16,16,24,16,16,17,0,0,23,16,0,0,2,16,1,3,17,1,16,16,18,19,
0,0,0,49,1,1,2,16,48,17,6,17,17,0,0,16,16,16,119,16,19,0,2,17,16,17,17,17,26,16,
50,17,49,0,19,0,0,0,2,16,17,49,17,129,17,16,2,16,3,16,0,0,1,6,2,16,17,16,1,17,
17,16,2,17,0,17,1,0,0,0,18,17,49,24,17,17,129,16,6,16,113,49,3,16,0,0,98,17,17,
17,16,49,0,96,9,16,16,0,0,1,49,0,18,17,17,17,17,17,16,1,2,81,1,0,32,0,16,16,2,
17,161,129,1,0,48,0,4,16,0,48,16,0,49,1,2,17,17,17,49,17,0,0,16,16,0,16,16,0,0,
48,2,80,17,17,17,19,17,48,0,16,16,0,0,0,17,6,2,32,48,32,6,0,34,32,98,2,34,34,38,
2,34,34,34,21,81,21,85,81,21,81,86,85,21,17,17,21,81,17,27,17,17,16,19,0,48,1,0,
16,0,0,0,48,0,0,2,19,17,49,0,17,1,17,2,22,1,0,1,1,48,0,2,16,17,0,0,48,0,49,2,80,
6,0,0,16,1,48,2,16,48,0,0,16,16,112,50,16,0,0,1,0,16,16,148,17,1,49,0,0,49,0,28,
16,0,0,48,3,0,0,2,16,0,3,1,0,0,17,18,16,3,0,0,0,0,0,50,17,113,1,3,0,0,48,2,16,0,
0,0,0,6,0,2,16,0,0,0,0,48,0,18,16,0,0,0,0,0,0,2,17,0,0,0,0,0,3,2,208,0,3,0,48,0,
0,4,17,112,48,19,16,0,0,18,17,0,0,0,0,0,16,100,16,17,9,1,1,16,1,2,22,0,0,0,0,0,
0,2,16,1,0,16,3,0,48,0,16,0,0,0,0,224,0,4,16,48,3,1,0,0,16,50,16,3,0,48,0,3,0,
52,16,16,1,16,0,0,0,2,16,16,0,0,2,2,1,4,68,66,32,34,36,34,34,36,34,36,34,34,68,
68,68,68
, // 13, scifi door
3,2,4,9,0,74,200,10,75,41,160,48,120,17,34,50,68,68,68,68,68,68,68,68,70,102,
102,102,102,102,102,102,85,85,153,85,85,153,85,85,85,136,136,136,136,136,136,
136,149,153,153,147,147,57,55,57,125,238,88,135,127,85,119,119,51,147,51,51,51,
55,147,55,215,125,215,119,119,119,119,119,68,68,68,68,68,68,68,70,102,102,102,
102,102,102,102,102,0,0,0,0,0,0,0,5,2,34,34,34,34,34,34,34,53,92,65,85,19,48,0,
81,32,11,160,0,1,18,34,1,48,83,64,0,83,16,0,1,34,1,178,34,1,2,34,33,53,84,65,0,
4,49,80,1,0,11,160,34,43,16,2,1,52,196,49,85,28,51,51,161,26,186,16,0,11,17,17,
161,51,51,51,51,51,51,49,16,1,17,17,17,17,17,17,16,53,85,49,85,19,21,0,34,32,0,
16,0,1,2,34,1,53,5,48,0,83,16,34,34,34,32,18,34,1,2,34,17,49,85,49,0,4,16,34,34,
34,32,16,34,43,2,32,161,51,68,49,85,28,16,34,34,34,32,16,0,11,0,17,0,51,67,51,
51,51,16,34,34,34,32,17,17,17,1,16,34,52,49,49,17,19,16,34,34,34,32,16,0,1,0,17,
2,51,5,48,0,84,16,34,34,34,32,18,34,10,2,1,17,48,5,49,5,84,16,34,34,34,32,16,32,
10,2,32,161,48,85,49,21,28,53,2,34,34,0,16,0,11,16,34,17,49,19,67,68,195,17,80,
177,16,1,161,171,177,0,0,177,51,52,17,51,67,51,17,16,1,26,0,17,161,17,17,16,53,
92,48,85,196,48,0,1,32,11,18,0,186,18,34,1,48,83,48,5,52,48,0,33,34,1,18,32,27,
18,34,33,53,84,49,85,68,17,80,1,0,11,16,0,186,0,2,1,52,196,51,76,67,19,51,161,
26,186,17,171,161,1,17,161,0,0,0,0,0,0,0,0,2,34,34,34,34,34,34,34,68,68,68,68,
68,68,68,102,102,102,102,102,102,102,102,102,51,147,51,51,51,55,147,125,125,119,
119,119,119,119,119,119,149,153,153,147,147,121,55,62,119,238,88,135,127,85,119,
119,85,85,153,85,85,153,85,85,152,136,136,136,136,136,136,136,68,68,68,68,68,68,
68,68,102,102,102,102,102,102,102,102
, // 14, concrete wall, tiles with skyscraper window
5,4,3,2,6,44,131,66,50,51,45,81,132,28,138,1,1,97,81,193,39,32,17,17,16,44,16,
21,17,33,5,17,17,97,17,17,23,49,17,0,17,33,17,0,16,33,21,28,17,97,17,81,19,48,
17,17,17,34,16,1,16,33,1,17,9,33,17,18,47,50,18,130,255,248,34,81,18,97,0,17,
153,55,50,34,34,34,34,34,34,34,34,34,34,147,49,17,63,114,68,68,68,68,0,64,68,4,
4,64,68,24,34,35,0,82,64,0,1,0,16,0,0,1,0,16,0,46,16,0,0,18,64,16,1,17,0,0,13,0,
0,1,0,31,16,1,18,59,0,16,0,0,0,0,0,0,0,0,0,35,17,17,115,59,65,17,0,0,0,0,0,0,0,
0,16,19,183,51,161,34,17,16,0,0,0,0,0,0,0,0,0,35,17,17,0,33,16,0,0,0,0,0,0,0,0,
4,208,19,17,17,17,18,64,0,0,0,0,0,0,0,0,1,0,19,18,17,17,41,65,16,0,0,0,0,0,0,0,
0,0,35,17,33,17,19,0,16,0,0,0,0,0,0,0,16,0,30,17,17,17,39,65,0,0,0,0,0,0,0,0,0,
16,50,17,21,17,19,0,21,17,0,0,0,0,0,0,0,0,34,17,17,17,43,64,0,0,0,0,0,0,0,0,29,
0,35,18,34,51,51,65,21,1,0,0,0,0,0,0,0,0,51,49,34,1,55,64,0,0,0,0,0,0,0,0,0,4,
27,17,16,16,62,64,16,1,0,0,0,0,0,0,0,0,54,17,17,81,7,64,0,160,0,1,0,0,0,0,1,0,
34,17,17,17,30,0,16,16,16,0,0,0,0,0,1,0,19,17,17,16,38,64,0,0,160,16,0,0,0,208,
16,209,34,17,33,17,147,0,16,17,0,0,16,0,16,0,1,0,50,17,17,16,35,64,0,0,0,0,0,0,
0,16,16,16,49,17,17,33,111,64,0,64,0,0,4,68,0,0,0,4,98,17,34,40,50,34,57,41,34,
99,35,50,34,18,33,33,129,0,4,50,129,17,1,193,50,34,34,34,143,35,34,34,35,51,39,
85,113,17,17,22,33,17,1,16,34,1,1,161,22,17,1,21,49,17,17,30,17,17,17,16,34,17,
16,16,18,10,17,16,97,12,17,22,32,17,0,17,145,17,16,0,144,1,17
, // 15, computer tech wall
52,59,53,220,38,141,54,143,75,76,77,61,74,5,66,95,48,0,0,0,10,17,0,0,0,0,38,0,0,
0,0,51,16,0,2,1,17,17,0,0,16,0,38,0,0,0,0,3,10,0,0,0,17,1,17,17,17,16,34,0,1,0,
0,0,7,0,0,0,17,49,81,33,33,32,34,48,1,48,0,0,2,0,0,0,17,1,1,17,17,16,38,0,0,0,0,
0,0,0,0,83,0,17,1,33,33,35,18,0,0,80,0,0,0,0,3,0,0,17,17,17,17,16,2,83,2,0,7,0,
0,85,19,19,3,17,17,33,33,32,2,17,0,32,0,0,0,0,48,0,1,16,1,0,16,0,34,17,17,0,0,0,
0,1,2,16,0,24,0,80,1,0,82,1,12,2,0,0,16,16,5,16,0,1,1,0,9,17,34,17,30,18,32,8,
16,0,96,0,0,17,17,17,17,17,34,80,85,16,32,1,0,2,2,0,0,17,68,68,68,68,34,5,85,1,
39,0,0,6,0,176,0,17,68,68,68,68,34,85,5,1,114,3,0,6,0,0,10,25,68,68,68,68,38,11,
85,1,114,3,0,6,0,0,0,17,68,68,68,68,34,0,48,1,2,3,2,6,0,0,0,17,68,68,68,68,34,0,
0,1,2,0,0,2,112,0,0,17,68,68,68,68,38,0,0,1,2,112,0,0,7,112,80,17,68,68,68,68,
38,0,0,21,34,0,0,16,32,0,0,19,34,34,34,34,34,5,0,80,34,177,17,227,0,53,1,17,16,
3,1,1,50,19,51,224,32,24,128,17,32,16,1,16,1,0,16,16,2,51,28,50,32,8,16,1,0,80,
0,21,0,0,16,1,146,51,137,6,0,1,0,1,16,0,17,17,0,0,17,16,34,17,144,32,0,3,0,0,17,
0,1,1,13,240,3,17,35,24,0,0,3,3,0,0,81,16,0,1,15,240,51,0,35,0,3,0,0,1,48,0,0,
195,0,17,0,0,5,49,34,0,3,0,160,3,0,0,5,48,206,17,4,208,109,0,2,48,0,112,2,3,57,
0,0,0,1,17,13,208,221,51,18,17,0,176,3,3,19,0,5,16,0,1,3,0,49,19,50,17,1,0,51,1,
144,0,48,0,0,17,0,0,0,48,34,49,48,0,0,3,48,0,0,0,10,17,0,0,0,0,38,0,0,0,0,51
};

SFG_PROGRAM_MEMORY uint8_t SFG_itemSprites[13 * SFG_TEXTURE_STORE_SIZE] =
{
// 0, barrel
175,6,106,29,100,84,7,92,2,43,10,11,46,4,28,200,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,255,0,0,0,0,0,0,
15,240,0,0,0,0,0,0,255,248,255,255,255,255,255,255,248,143,255,255,255,255,255,
255,143,248,248,255,255,143,136,136,130,136,136,136,136,136,136,136,143,168,66,
136,136,137,136,146,146,130,153,153,146,146,143,66,168,170,226,34,130,41,73,41,
66,137,34,34,153,41,146,226,168,186,34,34,155,149,187,186,162,170,84,153,34,146,
36,158,170,234,34,233,229,93,170,171,85,91,43,78,68,66,146,36,187,121,228,237,
69,90,181,85,85,93,91,132,77,66,36,73,158,89,18,68,69,187,85,183,119,122,181,
187,68,66,68,18,222,93,18,68,77,161,197,28,119,119,170,26,94,34,36,18,213,93,
193,20,21,177,17,81,87,119,123,183,84,52,44,20,222,93,17,17,23,17,17,21,119,117,
17,81,65,35,65,28,213,93,49,17,26,21,186,167,119,225,17,17,113,34,49,49,222,93,
60,19,106,97,106,119,115,49,97,193,118,35,67,19,222,93,18,102,101,102,102,103,
117,86,22,54,113,50,35,18,213,93,18,198,99,118,102,119,119,119,119,183,54,99,38,
25,190,93,102,102,195,166,203,119,119,123,190,101,108,50,51,99,181,93,54,54,51,
117,22,19,87,187,229,53,51,34,33,28,222,125,51,51,51,58,49,28,17,49,21,227,19,
19,51,17,158,185,195,51,51,49,174,81,17,85,117,17,17,193,28,76,155,186,51,195,
51,17,61,91,187,181,49,52,49,28,20,28,154,170,25,236,30,49,49,17,49,193,17,17,
28,20,19,28,184,170,25,225,193,17,17,17,17,193,17,17,17,20,19,30,184,138,60,238,
236,193,193,193,17,195,65,17,17,225,68,62,175,250,34,34,34,34,34,36,225,228,34,
34,34,34,34,36,175,251,0,0,0,0,0,0,4,64,0,0,0,0,0,0,191,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 1, health
175,7,5,6,2,94,0,69,4,53,71,93,3,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,68,68,64,0,4,68,68,64,0,0,0,0,0,0,0,4,17,17,20,68,65,17,17,20,
0,0,0,0,0,0,0,65,18,35,49,17,19,50,35,56,64,0,0,0,0,0,4,17,55,119,119,119,119,
119,119,50,70,0,0,0,0,0,4,34,119,41,41,41,41,41,39,114,134,0,0,0,0,0,4,34,114,
146,146,146,146,146,149,115,38,0,0,0,0,0,68,19,161,17,17,17,17,17,53,163,54,0,0,
0,0,4,34,35,163,17,17,17,17,17,21,163,38,0,0,0,0,4,108,41,161,21,85,85,85,177,
53,162,134,0,0,0,0,4,4,34,163,21,85,85,85,177,21,162,70,0,0,0,0,4,4,194,161,49,
49,91,17,49,53,168,96,0,0,0,0,4,0,66,163,19,17,91,19,19,21,168,96,0,0,0,0,4,0,
66,161,49,49,91,17,49,53,168,96,0,0,0,0,4,4,194,163,19,17,91,19,19,21,168,96,0,
0,0,0,4,4,34,161,21,85,85,85,177,53,162,70,0,0,0,0,4,108,41,163,21,85,85,85,177,
21,162,134,0,0,0,0,6,34,35,161,17,17,17,17,17,53,163,38,0,0,0,0,0,102,19,163,17,
17,17,17,17,21,163,54,0,0,0,0,0,4,34,121,41,41,41,41,41,37,115,38,0,0,0,0,0,4,
34,119,146,146,146,146,146,151,114,134,0,0,0,0,0,4,19,55,119,119,119,119,119,
119,50,70,0,0,0,0,0,0,98,50,35,40,136,130,50,35,40,96,0,0,0,0,0,0,6,136,136,134,
102,104,136,136,134,0,0,0,0,0,0,0,0,102,102,96,0,6,102,102,96,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 2, ammo: bullets
175,124,168,125,2,5,21,194,19,17,112,113,190,3,4,23,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,170,170,170,0,0,0,0,0,0,0,0,0,0,0,0,170,136,72,218,
0,0,0,0,0,2,34,34,34,34,32,10,104,102,214,73,32,0,0,0,0,34,119,119,119,119,34,
36,68,68,68,71,32,0,0,0,0,33,17,17,17,25,183,119,238,231,119,23,32,0,0,0,0,35,
17,17,20,68,183,119,238,231,119,23,32,0,0,0,0,35,17,20,72,132,183,119,238,231,
119,23,32,0,0,0,0,35,17,70,134,109,179,51,255,243,51,49,32,0,0,0,0,35,17,20,74,
170,179,119,85,81,17,49,32,0,0,0,0,35,17,26,168,132,179,113,85,81,17,49,32,0,0,
0,0,35,17,70,134,109,179,113,92,193,17,49,32,0,0,0,0,35,17,26,150,245,179,17,
204,81,17,49,32,0,0,0,0,35,17,20,137,153,179,17,197,81,17,49,32,0,0,0,0,35,17,
70,134,109,179,17,92,81,23,23,32,0,0,0,0,35,17,26,137,153,179,17,204,199,68,68,
66,0,0,0,0,35,17,20,152,132,179,17,85,68,136,72,66,0,0,0,0,35,17,70,134,109,179,
17,84,104,102,214,210,0,0,0,0,35,17,26,150,245,179,17,204,73,255,95,82,0,0,0,0,
35,17,20,152,132,179,17,85,193,153,154,162,0,0,0,0,35,17,70,134,109,179,17,92,
193,20,68,210,0,0,0,0,35,17,26,150,245,179,17,197,193,20,246,130,0,0,0,0,35,17,
72,137,153,179,17,197,81,20,93,66,0,0,0,0,35,17,26,152,132,179,17,92,193,25,246,
130,0,0,0,0,35,17,166,134,109,179,17,92,81,17,150,130,0,0,0,0,2,34,42,166,245,
179,17,204,193,17,152,32,0,0,0,0,0,0,0,2,34,179,17,85,81,17,58,32,0,0,0,0,0,0,0,
0,0,35,17,85,81,17,49,32,0,0,0,0,0,0,0,0,0,34,34,153,146,34,49,32,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,34,32,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0
, // 3, ammo: rockets
175,1,117,120,30,3,33,82,5,98,101,102,2,178,174,4,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,204,16,0,0,0,0,0,0,0,0,0,
0,0,0,17,26,197,81,17,17,17,16,0,0,0,0,0,0,0,17,204,202,85,92,113,119,23,16,0,0,
0,0,0,0,1,85,197,90,85,204,119,17,119,16,0,0,3,51,0,0,1,170,85,90,85,92,119,119,
119,16,0,3,59,47,51,0,1,85,197,90,85,204,124,119,199,193,0,59,187,42,169,51,53,
170,92,90,85,87,246,31,31,193,3,187,238,218,153,157,149,68,37,36,42,39,135,17,
17,17,0,62,238,217,153,51,53,85,82,36,34,167,136,97,17,16,0,3,222,217,211,0,6,
68,34,36,42,247,136,246,17,96,0,3,221,221,211,0,6,68,85,36,34,247,136,246,225,
96,0,59,187,42,169,51,53,68,133,36,42,39,136,246,209,96,3,187,238,218,153,157,
149,68,40,36,34,167,136,246,145,96,0,62,238,217,153,51,53,68,85,36,42,39,136,
246,225,96,0,3,222,217,211,0,6,68,130,36,34,167,136,246,209,96,0,3,221,221,211,
51,54,68,85,36,42,247,136,246,145,96,0,59,187,42,169,51,53,68,85,36,34,247,136,
246,225,96,3,187,238,218,153,157,149,68,34,36,42,39,136,246,209,96,0,62,238,217,
153,51,53,68,85,36,34,167,136,118,145,96,0,3,62,217,51,59,230,68,34,36,42,39,
136,201,153,16,0,0,3,51,0,50,182,68,34,36,34,167,135,106,153,16,0,0,0,0,0,3,54,
85,82,36,42,247,140,250,169,145,0,0,0,0,0,0,6,136,37,36,34,247,134,34,221,209,0,
0,0,0,0,0,0,102,72,68,42,39,134,187,238,236,0,0,0,0,0,0,0,0,102,104,34,166,102,
203,238,192,0,0,0,0,0,0,0,0,0,6,68,105,209,203,190,192,0,0,0,0,0,0,0,0,0,0,102,
1,145,28,188,0,0,0,0,0,0,0,0,0,0,0,0,0,17,16,192,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 4, ammo: plasma
175,6,0,2,4,142,206,5,7,62,205,115,207,116,1,201,0,0,0,0,35,51,51,51,51,50,0,3,
48,0,0,0,0,0,0,3,57,145,145,145,153,147,32,54,99,0,0,0,0,0,0,51,149,34,34,34,34,
52,34,42,162,34,34,32,0,0,2,57,50,143,255,34,243,34,238,39,66,238,62,32,0,0,3,
149,40,252,106,175,84,51,51,55,67,62,67,32,0,0,3,146,2,198,170,219,87,68,67,67,
52,67,116,32,0,0,3,146,2,198,170,221,87,68,67,68,68,67,116,32,0,0,3,146,2,198,
106,219,81,119,116,55,119,116,23,32,0,0,3,146,0,54,102,106,81,119,119,67,51,51,
52,32,0,0,3,146,0,251,187,187,184,17,17,17,17,23,67,32,0,0,3,50,2,106,221,219,
88,17,145,25,27,182,132,32,0,0,2,66,2,198,170,173,88,17,81,21,27,172,132,32,0,0,
3,50,2,198,106,171,88,17,81,89,27,172,132,32,0,0,2,66,0,54,102,106,88,153,85,
145,26,204,132,32,0,0,3,50,0,251,187,187,184,149,17,17,17,23,132,32,0,0,2,66,2,
106,221,219,88,17,25,25,17,23,132,32,0,0,3,50,2,198,170,221,88,21,21,21,17,23,
132,32,0,0,2,66,2,198,106,219,88,21,21,21,17,23,132,32,0,0,3,50,0,54,102,106,88,
25,21,81,17,23,132,32,0,0,2,66,0,251,187,187,184,17,17,25,17,23,132,32,0,0,3,50,
2,106,221,219,88,17,81,21,17,23,132,32,0,0,2,66,2,198,170,221,88,17,81,81,17,23,
132,32,0,0,3,146,2,198,106,219,88,17,85,21,17,23,132,32,0,0,3,146,0,54,102,106,
88,149,17,17,17,23,132,32,0,0,3,146,0,251,187,187,184,81,145,17,20,23,132,32,0,
0,3,146,2,106,221,219,88,17,81,81,23,23,132,32,0,0,3,146,2,198,170,221,88,17,81,
149,20,23,132,32,0,0,3,149,34,198,106,170,88,17,85,17,17,23,67,32,0,0,2,52,62,
54,102,106,88,17,17,51,51,51,52,32,0,0,0,35,69,46,238,238,247,119,115,71,119,
116,116,32,0,0,0,2,52,65,65,65,67,51,34,35,50,34,34,32,0,0,0,0,34,34,34,34,34,
34,0,2,32,0,0,0
, // 5, tree
175,0,17,115,5,3,2,196,4,114,20,38,22,96,37,21,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,51,0,0,0,1,17,0,0,0,3,54,32,0,0,0,3,51,51,0,17,26,72,0,0,0,
59,187,178,0,0,0,0,3,51,49,72,100,72,0,0,3,187,238,113,0,0,0,0,38,98,49,68,100,
72,0,0,6,190,231,57,16,0,0,34,187,238,33,68,100,72,0,0,34,190,119,57,16,0,0,43,
119,119,114,100,101,72,0,2,194,231,115,145,16,2,32,1,23,119,114,100,70,136,2,34,
194,231,57,145,0,44,16,51,49,19,114,68,134,85,2,204,207,147,153,145,2,193,3,51,
27,51,50,72,100,72,0,17,204,169,34,209,26,16,0,1,179,50,33,70,68,72,0,0,17,207,
82,37,173,16,0,0,17,38,98,86,164,72,0,0,0,17,26,85,33,98,0,1,170,85,82,72,106,
72,0,0,0,0,1,90,93,22,17,31,165,85,34,68,134,88,0,51,102,0,29,42,166,221,42,250,
85,34,33,68,72,101,3,187,183,17,204,246,250,82,255,85,82,45,210,100,70,72,3,190,
121,111,245,95,207,255,165,85,45,17,18,100,70,72,59,183,115,146,93,223,204,165,
85,45,17,119,114,68,72,101,59,231,115,146,209,165,44,109,17,29,42,17,50,68,134,
88,110,119,57,157,218,93,198,209,17,1,29,165,18,72,106,72,30,115,57,157,170,34,
198,17,145,0,49,37,82,86,164,72,1,51,153,37,93,28,81,41,16,3,49,210,82,70,68,72,
0,25,34,121,17,28,33,17,0,0,51,29,34,72,100,72,0,1,119,57,16,26,33,0,0,3,54,177,
34,68,134,85,0,1,115,145,16,21,16,0,3,54,107,18,82,100,70,136,0,0,17,17,2,25,18,
0,0,43,231,21,82,100,101,72,0,0,0,0,33,123,113,32,0,2,35,49,82,68,100,72,0,0,0,
0,27,238,121,16,0,0,49,17,18,68,100,72,0,0,0,0,30,231,57,16,0,51,51,51,50,72,
100,72,0,0,0,0,30,115,145,16,0,0,51,48,0,17,26,72,0,0,0,0,1,57,17,0,0,3,48,0,0,
0,1,17,0,0,0,0,0,17,16,0,0,0,0,0,0,0,0,0
, // 6, finish
175,3,4,0,5,1,2,6,50,7,198,209,48,42,44,49,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,51,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,147,3,51,51,51,51,51,
51,51,51,51,51,51,51,51,54,115,3,153,153,153,34,153,153,146,41,153,153,34,153,
153,153,115,3,119,119,119,34,119,119,114,39,119,119,34,119,119,119,67,3,68,68,
71,119,119,119,119,119,119,119,119,119,119,119,67,3,68,68,71,34,34,34,34,34,34,
34,34,34,34,34,67,53,85,85,71,40,136,136,136,136,136,136,136,136,129,68,115,49,
170,165,36,43,204,204,95,102,102,102,102,102,97,71,115,49,170,165,36,43,204,197,
246,102,102,102,102,102,97,65,67,49,161,165,36,43,187,255,216,136,136,136,136,
136,129,65,67,49,17,21,36,43,187,253,136,136,136,136,136,136,129,65,67,49,161,
165,36,43,187,253,136,136,136,136,136,136,129,65,67,49,26,21,36,43,102,129,17,
17,17,17,17,17,17,65,67,49,161,165,18,43,102,129,17,17,17,17,17,17,17,65,67,49,
17,21,18,43,102,129,17,17,17,17,17,17,17,65,67,49,170,165,18,43,17,34,34,34,34,
34,34,34,33,65,67,49,17,21,18,43,17,34,34,34,34,34,34,34,33,65,67,49,161,21,18,
43,18,68,68,68,68,68,68,68,65,65,67,49,170,165,142,43,18,68,68,68,68,68,68,68,
65,65,67,49,161,21,142,43,18,119,119,119,119,119,119,119,113,85,83,53,85,85,142,
38,18,68,68,68,68,68,68,68,65,245,83,3,102,102,97,34,34,34,34,34,34,34,34,34,33,
255,99,3,102,102,97,85,85,85,85,85,85,85,85,85,85,85,99,3,102,102,102,17,102,
102,97,22,102,102,17,102,102,102,99,3,85,85,85,17,85,85,81,21,85,85,17,85,85,86,
99,3,51,51,51,51,51,51,51,51,51,51,51,51,51,53,83,0,0,0,0,0,0,0,0,0,0,0,0,0,0,3,
83,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,51,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0
, // 7, teleport
175,0,151,6,3,143,5,4,134,53,55,2,127,63,45,54,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,17,16,0,0,1,17,0,0,1,17,17,16,0,0,0,1,183,177,0,0,27,123,16,0,18,33,18,
33,0,0,0,1,86,113,0,0,21,103,16,0,18,123,27,177,0,0,0,1,86,113,0,0,21,103,17,17,
18,55,183,113,0,1,17,17,68,65,176,11,20,68,67,51,123,51,182,97,0,20,51,51,51,51,
17,17,51,51,54,37,17,99,118,97,0,19,100,68,34,86,51,51,101,34,41,204,145,68,71,
113,0,19,68,68,34,34,34,37,82,34,124,136,155,68,68,65,0,19,221,37,82,40,130,34,
130,85,142,232,235,151,116,65,0,19,68,40,130,85,85,88,200,204,206,233,235,244,
75,177,0,19,69,172,205,165,34,34,85,136,204,136,228,254,70,116,0,19,68,172,202,
170,221,34,85,34,233,153,148,249,70,116,0,19,74,172,205,170,210,34,85,34,238,
153,151,249,70,116,0,19,74,170,221,170,210,34,85,37,140,136,151,249,118,116,0,
19,173,221,210,85,34,37,88,136,200,233,151,249,230,116,0,19,77,221,210,88,34,85,
136,136,200,153,158,249,230,116,0,19,68,221,221,218,34,34,85,37,136,136,158,249,
230,116,0,19,69,221,221,218,210,34,130,34,156,201,151,249,118,116,0,19,68,90,
170,170,173,34,130,34,156,201,151,249,118,116,0,19,170,74,221,213,130,37,85,136,
140,204,151,249,70,116,0,19,90,88,34,37,82,89,137,156,204,238,158,244,68,177,0,
19,165,40,34,34,130,37,130,85,94,232,155,246,51,49,0,19,68,68,34,34,34,34,85,34,
200,85,155,147,54,49,0,19,100,68,34,86,51,51,101,34,44,153,155,99,118,97,0,20,
51,51,51,51,17,17,51,51,54,85,113,115,118,97,0,1,17,17,68,65,176,11,20,68,67,51,
228,99,182,97,0,0,0,1,86,113,0,0,21,103,17,17,23,54,182,113,0,0,0,1,86,113,0,0,
21,103,16,0,18,107,23,113,0,0,0,1,183,177,0,0,27,123,16,0,18,33,18,33,0,0,0,0,
17,16,0,0,1,17,0,0,1,17,17,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 8, computer terminal
175,0,1,85,5,4,23,6,9,26,80,65,2,84,161,194,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,17,0,1,17,17,17,17,17,17,0,1,16,0,0,0,1,116,0,22,101,101,102,102,102,
101,16,1,65,0,0,238,225,117,1,99,53,53,51,51,51,53,187,27,75,17,17,238,17,69,22,
60,129,17,17,17,17,139,83,102,74,140,158,233,154,69,22,56,148,68,68,68,73,21,51,
54,74,142,238,51,25,69,22,49,66,34,34,34,36,19,68,54,90,238,141,61,169,69,21,
177,66,242,242,34,36,19,119,54,202,131,51,218,121,69,22,49,66,242,34,34,36,31,
119,54,90,141,216,167,121,69,22,49,66,242,242,34,36,131,119,54,202,17,138,119,
73,69,22,49,66,242,34,34,36,207,119,54,90,87,119,68,89,69,22,49,66,242,34,34,36,
195,119,54,90,148,68,85,89,69,22,49,66,34,34,34,36,195,119,54,90,181,85,85,89,
69,22,49,79,255,255,255,244,195,119,54,202,185,153,149,89,69,22,49,66,34,34,34,
36,195,119,54,90,185,153,153,153,69,22,49,66,34,34,34,36,140,119,54,90,17,138,
187,153,85,22,49,66,34,34,34,36,19,119,54,202,141,216,171,187,69,21,177,66,34,
34,34,36,28,119,54,90,131,51,218,187,85,22,49,66,34,34,34,36,19,68,54,90,133,85,
93,171,149,22,56,148,68,68,68,73,21,51,54,74,141,221,93,27,89,22,60,129,17,17,
17,17,139,83,54,74,136,136,136,138,149,1,51,60,60,51,51,51,53,187,27,75,17,17,
17,17,153,0,24,140,140,136,136,136,136,16,1,65,0,0,0,1,153,0,1,17,17,17,17,17,
17,0,1,16,0,0,0,1,187,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0
, // 9, column
175,2,4,3,5,48,6,63,51,7,81,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,65,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,20,65,0,0,0,0,0,0,0,0,0,0,0,0,0,0,20,73,16,0,0,0,0,0,0,0,
0,0,0,0,0,1,148,38,145,16,0,0,16,0,0,0,1,0,0,0,1,22,146,36,105,145,17,17,33,17,
17,17,23,17,17,17,25,150,98,36,102,67,119,119,41,121,150,102,103,119,118,121,54,
150,98,36,70,67,68,71,36,68,68,68,71,66,119,119,57,70,66,50,68,67,36,68,40,68,
68,67,34,66,36,68,52,36,66,34,34,35,51,51,40,34,50,40,34,35,50,34,52,36,66,50,
34,33,49,129,40,35,131,49,50,35,50,50,20,36,66,35,34,33,17,17,40,129,51,49,18,
51,50,131,18,36,34,51,51,49,17,24,33,19,51,17,50,56,51,131,18,34,66,50,51,49,26,
161,33,51,17,19,50,17,51,19,19,50,51,51,19,49,170,161,35,17,17,17,18,51,19,19,
19,50,35,49,19,17,161,168,34,130,35,56,34,34,33,131,17,34,33,49,50,53,85,85,37,
85,85,85,82,85,85,85,82,67,49,17,42,160,0,0,16,0,0,0,1,0,0,0,10,162,33,19,160,0,
0,0,0,0,0,0,0,0,0,0,0,10,33,26,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,17,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,17,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 10, ruin
175,5,4,3,50,153,2,75,12,6,13,21,240,17,74,83,0,0,0,0,0,0,0,5,0,0,0,9,25,56,18,
19,0,0,0,0,0,0,0,5,0,0,0,145,34,162,56,35,0,0,0,0,0,0,0,117,0,0,0,18,19,49,35,
50,0,0,0,0,119,87,116,69,119,87,82,25,34,51,50,162,0,0,0,5,84,84,84,84,84,85,69,
154,34,34,51,34,0,0,0,0,0,0,0,117,0,0,2,145,33,177,19,50,0,0,0,0,0,0,0,236,0,0,
153,17,17,18,34,51,0,0,0,0,0,0,0,204,0,9,155,17,177,17,17,19,0,0,0,0,0,0,0,236,
0,9,17,17,34,17,17,131,0,0,0,0,0,0,17,17,17,17,17,17,161,17,17,38,0,0,0,0,0,1,
17,35,51,17,177,17,17,17,19,99,0,0,0,0,0,1,19,35,57,33,17,17,18,177,35,102,0,0,
0,0,0,2,24,51,17,17,17,18,17,17,54,102,0,0,0,7,119,124,34,56,33,177,17,27,18,18,
102,214,0,0,0,119,84,92,19,50,33,18,33,17,17,40,99,51,0,0,119,117,0,4,19,51,161,
17,17,17,27,54,70,111,0,0,0,0,0,3,35,67,49,177,33,17,18,99,18,34,0,0,0,0,0,0,34,
51,52,50,34,162,38,49,161,130,0,0,0,0,0,0,65,34,130,68,50,18,131,34,17,33,0,0,0,
0,0,0,49,35,34,52,72,34,99,18,18,42,0,0,0,0,0,0,66,35,35,56,68,214,49,34,33,34,
0,0,7,119,71,71,193,51,50,51,34,70,50,18,162,33,0,0,0,85,84,84,65,35,131,50,132,
70,33,18,34,34,0,0,0,0,0,0,49,35,51,51,52,70,34,35,51,50,0,0,0,0,0,0,51,52,68,
67,68,46,131,51,51,51,0,0,0,0,0,0,244,68,221,77,221,34,102,102,54,109,0,0,0,0,0,
0,0,7,80,0,6,18,35,238,102,102,0,0,0,0,0,0,0,7,80,0,3,18,34,243,51,230,0,0,0,0,
0,84,68,71,84,119,117,146,34,40,51,246,0,0,0,0,0,0,68,85,85,85,67,18,35,35,34,
52,0,0,0,0,0,0,0,5,80,0,0,51,51,51,51,51,0,0,0,0,0,0,0,5,0,0,0,0,51,50,51,35
, // 11, lamp
175,0,49,31,7,106,27,28,107,3,4,105,1,2,26,34,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,220,16,0,0,0,0,0,0,0,0,0,0,0,0,0,1,220,
16,0,0,0,0,0,0,0,0,0,0,0,0,0,29,220,193,0,0,0,0,0,0,0,0,0,0,0,0,0,22,220,91,0,0,
0,0,0,0,0,0,17,0,1,16,85,103,146,88,85,0,0,0,0,0,0,1,33,0,22,133,51,103,146,232,
51,80,0,0,0,0,0,1,33,17,215,131,51,103,146,248,67,80,0,0,1,16,0,25,33,28,103,
132,68,103,146,248,68,59,187,177,26,33,17,153,33,28,119,132,68,231,98,232,68,55,
119,226,42,34,34,153,33,28,167,132,68,103,111,104,68,59,187,177,26,33,17,169,33,
17,167,131,51,119,98,136,67,80,0,0,1,16,0,26,33,0,23,85,51,119,111,230,51,80,0,
0,0,0,0,1,33,0,1,16,85,119,146,230,85,0,0,0,0,0,0,1,33,0,0,0,0,26,146,97,0,0,0,
0,0,0,0,0,17,0,0,0,0,26,162,241,0,0,0,0,0,0,0,0,0,0,0,0,0,1,175,16,0,0,0,0,0,0,
0,0,0,0,0,0,0,1,162,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 12, access card
175,0,21,45,44,23,46,2,19,34,69,22,68,60,131,8,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,17,17,17,17,17,17,0,0,0,0,0,0,0,0,1,
68,17,24,102,101,86,97,0,0,0,0,0,0,0,0,1,68,17,24,102,85,86,101,16,0,0,0,0,0,0,
0,1,67,17,24,101,85,86,101,183,16,0,0,0,0,0,0,1,51,17,24,85,85,102,59,178,16,0,
0,0,0,0,0,1,51,17,24,85,86,232,219,34,240,0,0,0,0,0,0,1,51,17,24,85,86,131,130,
34,240,0,0,0,0,0,0,1,51,17,24,85,102,52,114,34,16,0,0,0,0,0,0,1,54,17,24,86,99,
56,66,34,16,0,0,0,0,0,0,1,102,17,24,86,51,141,114,34,16,0,0,0,0,0,0,1,101,17,24,
102,51,211,114,34,16,0,0,0,0,0,0,1,101,17,24,99,51,55,114,34,16,0,0,0,0,0,0,1,
85,17,24,51,51,119,226,34,16,0,0,0,0,0,0,1,85,17,24,51,52,71,34,34,16,0,0,0,0,0,
0,1,85,17,24,51,52,119,130,34,16,0,0,0,0,0,0,1,85,17,24,51,68,228,114,34,16,0,0,
0,0,0,0,1,85,17,24,52,68,66,34,34,16,0,0,0,0,0,0,1,86,17,25,52,68,66,34,34,16,0,
0,0,0,0,0,1,86,17,23,68,68,34,34,34,16,0,0,0,0,0,0,1,102,17,23,68,68,34,34,34,
16,0,0,0,0,0,0,1,99,17,23,68,153,153,114,34,16,0,0,0,0,0,0,1,99,17,23,68,154,
170,114,34,16,0,0,0,0,0,0,1,51,17,23,68,154,204,114,34,16,0,0,0,0,0,0,1,51,17,
23,68,154,204,114,34,16,0,0,0,0,0,0,1,51,17,23,68,151,119,114,34,16,0,0,0,0,0,0,
1,51,17,23,66,130,34,130,34,16,0,0,0,0,0,0,1,51,17,23,66,130,34,130,34,16,0,0,0,
0,0,0,0,17,17,17,17,17,17,17,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0
};

SFG_PROGRAM_MEMORY uint8_t SFG_backgroundImages[3 * SFG_TEXTURE_STORE_SIZE] =
{
// 0, city
64,10,73,168,19,12,14,13,80,1,72,101,0,100,136,57,17,17,17,68,87,119,119,119,
118,8,0,32,0,46,34,154,17,17,20,85,84,85,119,118,102,96,0,130,130,2,34,154,17,
17,20,65,68,69,119,102,99,48,48,0,2,32,34,154,17,17,17,17,68,68,87,118,102,102,
104,0,2,14,34,154,17,17,17,20,69,69,51,51,0,8,0,224,2,32,34,154,17,17,17,20,85,
87,60,204,195,51,0,0,0,2,34,154,17,17,17,68,85,119,252,204,51,224,48,0,0,32,34,
154,17,17,17,68,85,119,51,51,48,3,0,0,130,34,34,154,17,17,68,68,85,119,119,102,
102,104,48,40,32,32,34,154,17,17,20,69,87,119,119,102,102,102,128,128,0,2,34,
154,17,17,17,69,87,51,51,51,0,8,48,40,32,32,34,154,17,17,17,20,87,60,252,252,
243,51,0,0,128,2,34,154,65,17,17,20,85,60,204,204,51,48,48,0,32,32,34,154,17,17,
17,68,85,51,63,51,48,51,8,130,130,2,34,154,17,17,20,69,103,181,85,86,102,0,56,0,
0,46,34,154,17,17,17,69,85,187,85,91,102,3,15,0,0,2,34,154,17,17,17,68,85,187,
51,51,0,8,8,0,0,32,34,154,17,17,17,68,77,187,51,51,48,0,142,0,0,2,34,154,17,17,
17,68,77,219,51,51,0,8,48,0,0,32,34,154,17,17,65,69,51,51,60,204,195,51,224,0,0,
2,34,154,17,17,68,69,63,239,204,204,51,48,224,0,0,32,34,154,17,17,71,69,51,51,
51,51,48,240,136,128,14,2,34,154,68,17,71,68,85,221,187,219,176,8,48,40,32,46,
34,154,17,17,20,68,69,93,219,211,48,0,0,0,0,2,34,154,17,17,68,68,77,219,221,211,
0,8,48,0,224,32,34,154,65,20,69,85,84,219,221,179,48,0,136,128,0,2,34,154,17,68,
69,85,119,123,221,219,182,104,8,32,0,32,34,154,17,17,68,69,87,107,221,187,102,
102,102,128,0,2,34,154,17,17,20,85,87,107,219,190,0,8,8,0,0,32,34,154,17,17,20,
84,86,107,182,102,102,3,224,0,2,2,34,154,17,17,20,68,71,103,118,110,0,8,0,40,34,
32,34,154,17,17,17,68,71,102,103,102,102,102,104,128,0,2,34,154
, // 1, reddish scifi interior
168,18,17,19,65,153,1,9,152,232,0,45,61,80,96,160,18,114,34,34,34,34,34,80,0,0,
0,0,0,0,0,0,18,114,34,34,34,34,34,80,0,0,0,0,0,0,0,0,0,215,34,34,34,34,34,80,0,
0,0,0,0,0,0,0,0,119,34,34,34,34,34,80,0,0,0,0,0,0,0,0,0,214,34,34,34,34,34,34,
32,0,0,0,0,0,0,0,0,118,34,34,34,34,34,34,36,69,0,0,0,0,0,0,18,118,17,17,17,17,
17,17,20,69,0,0,0,0,0,0,18,118,17,17,17,17,17,17,20,69,0,0,0,0,0,0,18,118,114,
34,34,34,34,34,116,69,0,0,0,0,0,0,18,118,130,34,34,38,0,0,0,0,0,0,0,0,0,0,18,
118,135,34,34,34,0,0,0,0,0,0,0,0,0,0,85,86,138,34,34,34,34,41,133,0,0,0,0,0,0,0,
0,86,138,17,17,17,17,25,133,85,0,0,0,0,0,0,0,86,138,17,17,17,27,25,132,70,0,0,0,
0,0,0,0,85,85,85,17,17,17,25,132,70,0,0,0,0,0,0,0,0,0,5,17,17,17,25,132,70,0,0,
0,0,0,0,0,0,0,5,17,27,17,25,132,70,0,0,0,0,0,0,0,6,128,17,17,17,17,25,132,70,0,
0,0,0,0,0,16,118,128,17,17,27,17,25,246,0,0,0,0,0,0,0,16,118,128,17,17,17,17,25,
132,0,0,0,0,0,0,0,16,118,128,51,51,51,51,57,246,0,0,0,0,0,0,0,16,118,130,51,51,
51,51,50,132,0,0,0,0,0,0,0,16,118,130,51,51,51,51,50,132,0,0,0,0,0,0,0,16,118,
131,51,51,51,51,51,132,70,0,0,0,0,0,0,0,0,5,60,51,51,51,51,36,70,0,0,0,0,0,0,0,
0,5,51,51,51,51,51,36,70,0,0,0,0,0,0,0,14,229,51,51,51,51,51,52,70,0,0,0,0,0,0,
85,85,51,195,51,51,51,51,52,64,0,0,0,0,0,0,18,114,17,17,17,17,17,17,18,64,0,0,0,
0,0,0,18,113,17,17,17,17,17,17,17,64,0,0,0,0,0,0,18,115,51,51,51,51,51,80,0,0,0,
0,0,0,0,0,18,115,51,51,51,51,51,80,0,0,0,0,0,0,0,0
, // 2, red city at night
171,1,175,90,172,174,173,25,27,30,91,103,0,0,0,0,113,23,113,17,51,68,68,102,102,
85,82,34,34,34,38,49,113,23,113,17,51,68,68,102,102,85,82,34,34,34,35,51,113,0,
0,0,0,0,0,0,68,69,85,19,4,101,166,17,113,0,0,4,68,102,102,85,85,81,17,19,4,101,
38,54,113,0,0,4,68,102,102,85,85,81,17,19,4,101,38,51,113,0,0,4,68,102,102,85,
85,82,33,19,4,101,34,49,113,0,0,0,0,0,0,0,4,69,81,19,4,101,42,17,119,17,17,17,
17,0,4,68,102,85,85,35,4,101,42,51,135,17,17,51,51,51,0,0,0,0,68,83,4,101,34,17,
152,113,17,59,51,48,0,4,68,102,85,19,4,101,34,83,152,113,17,51,51,48,0,4,68,17,
17,19,4,101,42,51,135,113,17,51,51,48,0,4,68,17,17,19,4,101,164,17,119,17,17,59,
51,48,0,4,68,102,82,19,4,101,165,17,119,17,17,51,51,51,0,0,0,0,69,85,85,34,37,
51,119,113,17,17,17,16,4,68,102,101,82,37,85,34,34,102,119,119,17,19,51,0,4,70,
102,85,34,37,85,34,34,34,119,119,17,17,17,0,4,70,101,85,34,37,85,34,34,34,119,
17,17,17,17,16,4,70,101,82,38,17,48,70,51,34,113,16,0,0,0,0,0,0,102,102,33,17,
48,67,50,38,17,16,176,176,0,4,68,102,101,82,33,17,48,17,51,51,113,16,0,0,0,4,68,
102,101,82,33,17,48,17,51,50,113,16,0,0,0,4,68,102,101,82,38,17,48,17,51,51,135,
17,17,17,17,16,0,0,0,82,34,34,34,35,50,34,136,113,17,17,17,19,0,0,70,85,85,34,
34,36,51,36,136,113,17,16,0,3,48,0,0,0,68,102,82,36,17,17,152,135,17,16,0,0,51,
0,4,68,85,85,34,33,17,22,153,135,17,16,0,0,0,0,0,0,68,102,82,37,17,17,153,135,
17,17,17,16,0,68,70,101,1,19,4,101,81,34,152,119,16,0,0,0,4,68,64,17,17,19,4,
101,34,36,152,119,16,0,0,0,0,68,70,101,1,19,4,101,35,51,135,119,16,0,0,0,0,4,70,
101,85,34,34,34,35,54,119,23,16,0,0,0,0,0,0,4,69,82,34,34,35,51
};

SFG_PROGRAM_MEMORY uint8_t SFG_weaponImages[6 * SFG_TEXTURE_STORE_SIZE] =
{
// 0, knife
175,5,4,2,0,3,6,1,61,83,85,60,63,77,40,7,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,68,68,0,0,0,0,0,0,0,0,0,0,0,0,0,4,97,163,64,0,0,0,0,0,0,0,0,0,0,0,0,4,106,169,
64,0,0,0,0,0,0,0,0,0,0,0,0,4,26,217,64,0,0,0,0,0,0,0,0,0,0,0,0,0,74,147,64,0,0,
0,0,0,0,0,0,0,0,5,85,85,57,147,116,0,0,0,0,0,0,0,0,5,85,86,108,136,61,153,116,0,
0,0,0,0,0,0,85,88,102,102,102,136,49,148,68,68,0,0,0,0,0,88,17,102,204,102,102,
136,54,227,83,52,0,0,0,0,85,34,17,102,102,111,246,24,118,114,34,34,0,0,0,85,17,
34,22,111,108,97,17,18,118,114,53,83,0,0,85,17,22,17,17,34,34,37,85,87,113,117,
51,83,0,5,17,22,24,130,34,17,17,24,27,37,114,51,51,83,0,81,22,98,187,184,17,102,
97,17,34,37,114,51,51,51,5,17,104,187,34,134,102,102,102,18,33,37,121,51,51,51,
5,22,139,130,40,24,140,204,129,34,17,37,121,115,115,119,81,17,85,82,82,33,28,97,
17,34,34,37,125,119,115,119,51,51,52,116,68,68,49,33,21,34,82,37,125,55,119,119,
0,0,0,0,0,0,3,65,35,43,226,37,125,151,71,119,0,0,0,0,0,0,0,3,126,35,69,37,58,
147,116,68,0,0,0,0,0,0,0,0,14,52,4,87,61,147,116,0,0,0,0,0,0,0,0,0,0,0,14,68,57,
147,64,0,0,0,0,0,0,0,0,0,0,0,0,0,74,147,64,0,0,0,0,0,0,0,0,0,0,0,0,4,26,217,64,
0,0,0,0,0,0,0,0,0,0,0,0,4,170,167,64,0,0,0,0,0,0,0,0,0,0,0,0,4,157,167,64,0,0,0,
0,0,0,0,0,0,0,0,0,0,68,68,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0
, // 1, shotgun
175,3,4,54,5,17,2,1,7,101,100,99,25,98,9,18,0,0,0,0,0,0,0,0,0,0,0,0,0,0,119,119,
0,0,0,0,0,0,0,0,0,0,0,0,102,103,91,170,0,0,0,0,0,0,0,0,0,0,1,22,67,45,186,153,0,
0,0,0,0,0,0,0,1,17,20,50,52,218,153,74,0,0,0,0,0,0,0,17,18,35,136,55,20,218,153,
75,0,0,0,0,0,17,17,68,72,131,51,71,20,91,169,170,0,0,0,0,17,52,51,136,51,35,136,
35,52,93,187,180,0,0,0,1,51,19,131,56,130,67,50,36,66,91,186,148,0,0,0,19,51,19,
136,136,50,36,17,71,18,93,186,154,0,0,0,24,134,56,131,52,68,17,194,39,18,85,219,
187,0,0,6,51,54,51,52,68,66,34,34,34,34,21,255,185,0,0,6,51,50,50,34,34,34,33,
17,17,23,28,93,169,0,0,87,114,34,98,17,17,17,17,17,17,23,28,85,169,0,9,188,231,
21,97,81,85,85,85,92,92,197,197,87,218,0,5,85,206,236,229,85,85,117,117,119,119,
119,199,119,221,0,0,7,204,193,17,17,17,17,17,17,17,17,252,87,218,0,5,85,204,202,
170,169,153,153,153,153,153,170,170,87,169,0,9,188,199,34,193,17,17,17,17,17,17,
23,17,85,169,0,0,87,115,50,35,51,136,51,52,68,34,39,17,91,153,0,0,6,136,134,56,
136,136,136,131,51,136,136,65,219,185,0,0,6,51,54,67,51,51,131,56,136,55,24,77,
186,187,0,0,0,98,38,35,68,68,51,34,35,55,20,45,169,148,0,0,0,100,68,100,34,34,
44,18,17,67,52,219,153,68,0,0,0,6,68,34,34,18,38,97,28,98,34,219,171,186,0,0,0,
0,102,17,68,33,17,108,17,103,17,91,219,171,0,0,0,0,0,102,102,20,68,34,33,103,22,
93,186,154,0,0,0,0,0,0,0,102,98,34,68,70,22,91,186,153,0,0,0,0,0,0,0,0,6,102,98,
44,38,93,186,153,0,0,0,0,0,0,0,0,0,0,6,102,34,101,219,153,0,0,0,0,0,0,0,0,0,0,0,
0,102,103,95,170,0,0,0,0,0,0,0,0,0,0,0,0,0,0,119,119,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0
, // 2, machine gun
175,2,4,3,1,5,6,0,60,59,50,53,61,62,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,20,68,64,0,0,0,0,0,0,0,0,0,0,0,17,34,51,17,20,0,
0,0,0,0,0,0,0,0,0,1,98,17,17,68,71,65,17,16,0,0,0,0,0,0,0,22,33,17,20,68,70,102,
35,167,0,0,0,0,0,0,0,22,33,20,68,97,22,98,50,228,0,0,0,0,0,0,0,118,52,70,17,102,
102,34,34,20,0,0,0,0,0,0,7,51,17,22,102,102,101,37,35,20,0,0,0,0,0,0,22,102,22,
34,82,140,178,34,49,20,0,0,0,0,0,1,82,34,51,50,34,43,178,51,161,68,0,0,0,0,17,
17,34,35,51,152,49,19,51,161,68,68,0,0,1,17,88,83,34,58,19,136,37,37,50,51,133,
85,0,0,22,150,134,99,35,147,50,34,85,85,34,51,34,85,0,0,121,24,56,209,35,51,50,
34,85,82,34,51,34,37,0,0,113,19,18,33,35,51,34,37,85,34,37,51,34,34,0,0,1,119,
49,49,147,38,101,85,85,34,85,51,130,34,0,0,0,0,119,113,147,50,34,86,102,86,98,
161,68,68,0,0,0,0,0,7,25,34,37,85,34,37,195,51,161,68,0,0,0,0,0,0,113,17,17,51,
53,92,131,51,49,20,0,0,0,0,0,0,7,19,20,65,17,17,19,50,51,20,0,0,0,0,0,0,0,118,
20,65,68,17,17,50,35,20,0,0,0,0,0,0,0,22,17,68,68,20,65,19,51,20,0,0,0,0,0,0,0,
22,33,17,68,68,65,17,51,23,0,0,0,0,0,0,0,7,98,49,17,20,119,68,68,64,0,0,0,0,0,0,
0,0,23,34,51,17,20,0,0,0,0,0,0,0,0,0,0,0,0,17,119,119,64,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 3, rocket launcher
175,33,7,39,115,2,0,5,117,116,4,37,114,3,36,108,0,0,0,0,0,0,0,0,0,0,0,0,13,209,
102,0,0,0,0,0,0,0,0,0,0,0,0,221,218,122,85,96,0,0,0,0,0,0,0,0,0,0,13,34,39,165,
51,96,0,0,0,0,0,0,0,0,0,0,5,170,170,165,51,96,0,0,0,0,0,0,0,0,0,0,5,93,221,221,
81,102,0,0,0,0,102,102,16,0,0,0,17,102,102,102,102,204,0,0,0,17,170,173,214,80,
17,17,136,137,198,204,193,206,0,0,93,122,213,81,17,17,136,136,137,68,193,236,
158,238,0,5,170,17,22,20,148,152,148,204,196,153,66,153,136,136,0,90,161,17,100,
68,76,68,76,153,153,137,40,136,136,136,5,163,17,17,73,153,201,153,201,136,139,
190,43,179,51,51,5,58,81,20,136,132,136,132,139,179,55,226,51,51,51,51,13,42,17,
24,187,180,68,68,178,34,34,162,34,34,34,35,215,33,17,72,187,119,114,36,34,34,35,
162,35,51,51,17,211,33,21,131,50,34,34,46,34,34,34,162,34,34,209,51,163,37,85,
131,51,51,34,46,34,34,34,114,34,45,51,119,163,37,85,232,51,51,51,62,51,51,51,
178,51,127,55,187,211,37,85,238,238,238,238,228,136,136,187,226,51,115,123,185,
215,49,81,20,68,68,73,148,153,152,142,146,235,175,251,153,13,58,17,28,68,68,68,
68,68,68,153,67,153,175,123,153,5,122,17,81,204,196,68,68,68,68,68,67,116,175,
123,137,5,119,85,85,97,204,204,204,204,68,68,71,36,223,123,185,0,87,117,85,81,
102,17,17,28,204,204,196,58,211,123,184,0,5,119,165,85,81,102,85,17,17,17,204,
167,211,119,187,0,0,81,122,213,85,85,22,101,85,17,17,74,31,119,187,0,0,0,17,170,
173,214,16,102,102,85,17,17,31,247,119,0,0,0,0,102,102,96,0,0,0,102,102,81,31,
119,119,0,0,0,0,0,0,0,0,0,0,0,0,102,23,123,183,0,0,0,0,0,0,0,0,0,0,0,0,0,109,
123,187,0,0,0,0,0,0,0,0,0,0,0,0,0,97,215,136,0,0,0,0,0,0,0,0,0,0,0,0,0,6,109,
255,0,0,0,0,0,0,0,0,0,0,0,0,0,0,6,102
, // 4, plasma gun
175,4,3,6,5,17,50,49,0,187,7,190,37,42,100,209,0,0,0,0,0,0,0,0,0,0,0,0,2,80,1,
17,0,0,0,0,0,0,0,0,0,0,37,0,44,149,26,163,0,0,0,0,0,0,0,2,80,2,198,80,134,106,
163,51,0,0,0,0,0,0,0,44,101,2,194,37,170,170,51,51,0,0,0,0,0,0,2,201,103,88,42,
170,163,51,21,86,0,0,0,0,119,0,2,194,37,170,74,51,52,85,102,17,0,0,0,34,68,112,
40,42,170,172,67,51,70,33,17,17,0,0,34,52,68,130,74,164,51,196,51,51,110,17,18,
85,0,2,51,66,136,170,67,51,68,67,51,52,81,17,37,51,0,42,52,119,170,68,51,51,51,
51,51,66,17,18,68,68,2,163,55,42,163,51,52,66,221,85,93,33,17,68,17,153,2,119,
119,118,102,102,98,33,17,18,17,18,20,73,187,187,42,52,17,17,17,17,17,17,17,17,
33,17,36,155,149,136,36,18,119,117,86,97,20,68,68,68,36,68,33,185,83,67,35,68,
65,39,150,150,22,22,145,105,22,145,27,181,51,20,35,68,17,39,185,185,185,185,187,
155,185,187,187,182,52,20,35,65,18,39,150,150,22,22,145,105,22,145,27,182,68,20,
33,34,119,117,86,97,20,68,68,68,36,68,33,185,97,33,36,17,19,51,51,51,51,51,51,
51,35,51,36,155,150,88,2,119,118,102,102,102,34,36,68,50,51,50,52,73,187,187,2,
17,39,33,17,17,17,18,101,101,85,99,51,68,17,153,0,34,18,117,34,34,17,17,17,17,
17,229,51,50,68,68,0,7,34,34,88,119,114,17,119,209,17,17,83,51,37,51,0,0,119,34,
39,136,247,47,34,45,17,17,110,51,50,85,0,0,0,119,119,128,136,85,119,34,114,33,
22,228,51,51,0,0,0,0,136,0,8,198,87,117,242,34,33,102,85,67,0,0,0,0,0,0,8,201,
101,136,85,82,34,34,34,85,0,0,0,0,0,0,0,140,152,8,201,136,82,34,34,34,0,0,0,0,0,
0,0,8,128,8,198,128,134,101,34,34,0,0,0,0,0,0,0,0,0,0,136,0,140,152,133,85,0,0,
0,0,0,0,0,0,0,0,0,0,8,128,8,136,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 5, solution
6,76,175,48,26,5,4,7,57,77,75,27,58,2,28,68,34,34,34,34,34,34,34,34,45,221,211,
34,35,51,50,51,34,34,34,51,51,34,34,34,55,7,119,50,48,6,67,21,34,34,51,69,86,51,
56,45,112,87,119,3,8,136,68,161,34,34,53,102,70,102,68,215,0,87,7,102,131,173,
133,170,34,34,48,182,68,70,103,112,0,231,0,100,134,108,218,90,34,34,48,96,181,
68,112,0,0,231,0,84,134,119,74,165,34,34,53,176,96,183,0,0,5,21,80,4,77,7,124,
170,34,51,53,70,102,112,0,5,105,145,21,0,70,0,7,202,51,85,104,136,71,0,0,97,17,
153,17,85,0,0,0,124,48,70,102,104,135,0,6,17,31,25,17,17,85,0,0,7,48,180,68,70,
103,0,81,17,17,17,153,17,25,80,0,0,48,101,182,68,71,0,81,17,26,17,249,145,17,23,
0,5,48,224,96,230,71,0,101,17,17,17,25,145,17,23,0,92,53,176,224,96,71,0,105,81,
250,241,241,153,17,23,0,202,51,54,70,181,64,0,97,150,144,154,15,153,17,23,0,198,
34,35,68,68,64,5,97,31,127,122,122,25,145,16,85,134,34,35,56,136,128,0,97,31,1,
15,10,17,145,23,0,134,51,51,68,68,64,0,97,26,145,144,159,17,25,23,0,138,53,102,
70,69,176,0,97,17,169,170,241,17,25,151,0,138,53,176,176,224,183,5,97,17,25,145,
17,17,17,151,0,88,48,224,96,101,183,5,81,17,26,153,241,17,17,112,0,5,53,101,229,
180,135,0,85,17,17,25,145,17,80,0,0,0,53,228,77,136,135,0,5,81,26,25,97,7,5,0,0,
6,51,68,136,132,71,0,0,85,81,17,151,0,0,80,0,76,34,51,133,70,70,112,0,85,5,16,
112,0,102,5,100,204,34,34,53,176,229,71,0,5,80,231,0,5,68,102,76,204,34,34,48,
224,182,68,112,0,85,231,0,6,72,68,204,197,34,34,48,182,68,68,71,112,5,87,0,100,
104,141,220,90,34,34,53,68,68,136,136,55,5,96,5,100,3,173,133,170,34,34,51,136,
136,51,51,35,112,102,100,136,64,136,132,161,34,34,34,51,51,34,34,34,54,102,68,
50,56,100,67,21,34,34,34,34,34,34,34,34,35,51,51,34,35,51,50,51
};

SFG_PROGRAM_MEMORY uint8_t SFG_effectSprites[4 * SFG_TEXTURE_STORE_SIZE] =
{
// 0, explostion
175,183,174,15,103,111,191,7,31,95,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,48,48,0,0,0,
0,0,0,0,0,0,0,0,3,3,3,3,19,3,0,0,0,0,0,0,0,0,0,48,48,16,51,49,19,48,48,0,0,0,0,
0,0,0,3,3,1,1,19,17,131,51,49,0,0,0,0,1,16,0,0,48,50,34,20,85,120,49,17,64,0,0,
0,0,65,1,0,0,34,17,69,87,117,17,33,16,0,0,0,0,4,16,16,4,18,17,21,87,85,18,33,16,
0,0,0,1,9,34,34,17,34,17,21,85,85,66,17,48,0,0,0,65,18,34,34,17,65,22,85,86,68,
65,19,51,0,0,48,17,34,34,33,18,65,68,70,68,68,102,67,51,0,1,19,49,34,34,33,18,
36,132,100,70,97,20,81,17,51,48,17,18,34,18,18,34,33,119,85,86,97,33,81,51,48,3,
51,18,33,17,34,33,37,119,85,100,97,18,35,51,0,48,49,34,18,34,33,102,87,119,117,
97,17,34,34,16,48,1,17,18,33,17,22,104,119,119,117,81,18,18,33,67,0,1,65,18,33,
20,71,119,119,119,119,132,65,18,34,64,48,0,69,81,17,86,69,88,119,119,133,81,17,
18,34,147,0,0,56,21,69,86,70,103,119,119,116,17,18,34,34,144,0,3,51,56,86,102,
102,120,118,101,120,81,34,34,35,0,0,48,51,17,85,102,102,103,70,102,136,85,34,34,
17,0,0,3,49,19,53,84,102,100,70,102,88,133,84,66,17,19,0,48,51,51,85,68,102,68,
102,102,101,97,33,68,19,17,48,3,3,49,81,36,84,17,70,101,82,33,18,33,67,51,0,0,
51,49,17,33,33,33,70,102,66,34,34,34,17,51,48,0,3,17,18,34,33,34,20,129,18,34,
34,34,34,51,0,0,49,17,34,33,17,17,33,65,34,34,17,18,34,19,48,0,1,18,33,17,17,17,
34,34,33,145,17,18,34,19,48,0,1,17,17,19,49,49,18,17,49,19,49,18,34,35,51,0,0,
17,51,51,17,49,18,19,3,19,51,17,17,36,3,0,3,51,51,48,48,3,17,16,48,17,3,51,17,
16,0,0,3,48,3,3,0,0,16,19,0,0,0,51,51,0,0,0,51,0,0,0,0,0,1,3,48,0,51,51,48,0,0
, // 1, fireball
175,103,183,7,191,111,254,31,95,180,173,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,17,0,16,0,0,0,0,
0,0,0,0,0,0,1,1,23,17,17,24,1,0,0,0,0,0,0,0,0,0,16,16,23,85,87,81,24,16,0,0,0,0,
0,0,0,1,8,17,119,115,55,119,17,129,16,0,0,0,0,0,0,0,129,23,119,51,51,55,113,24,
23,0,0,0,0,0,0,8,17,17,113,83,51,85,119,23,113,0,0,0,0,0,0,1,21,113,81,19,53,81,
17,119,129,0,0,0,0,0,0,17,19,53,17,19,50,17,17,113,16,16,0,0,0,0,1,33,83,49,17,
19,66,34,17,17,18,0,0,0,0,0,0,33,85,53,21,19,18,34,34,24,18,16,0,0,0,0,1,33,85,
85,84,68,20,34,35,66,98,0,0,0,0,0,0,33,83,53,20,68,51,51,51,70,34,16,0,0,0,0,1,
33,19,49,66,36,67,52,41,40,98,0,0,0,0,0,0,33,19,84,34,36,36,70,38,134,34,0,0,0,
0,0,0,8,21,81,18,36,66,98,102,166,32,0,0,0,0,0,0,2,21,81,18,68,68,38,134,105,33,
0,0,0,0,0,0,0,18,33,18,68,51,73,102,98,16,0,0,0,0,0,0,0,18,18,34,68,51,74,102,
102,16,0,0,0,0,0,0,0,1,17,18,34,36,38,38,129,0,0,0,0,0,0,0,0,0,8,134,38,34,34,
104,0,0,0,0,0,0,0,0,0,0,0,2,34,102,96,16,0,0,0,0,0,0,0,0,0,0,0,0,1,16,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0
, // 2, plasma
175,199,126,213,198,215,118,46,125,204,124,134,45,117,135,116,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,1,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,68,16,0,0,0,0,0,0,0,0,0,0,16,
1,17,20,17,16,16,16,0,0,0,0,0,0,0,17,18,7,113,20,17,119,1,1,0,0,0,0,0,0,1,17,65,
16,34,67,50,33,17,16,0,0,0,0,0,0,1,36,34,1,19,153,51,33,65,0,0,0,0,0,0,0,14,37,
34,32,51,57,163,35,50,16,0,0,0,0,0,0,114,37,82,2,17,51,170,51,34,17,0,0,0,0,0,0,
18,51,83,34,17,3,152,138,43,178,16,0,0,0,0,1,20,34,52,83,163,32,123,136,199,34,
17,0,0,0,0,1,17,18,35,51,51,34,18,34,194,36,68,0,0,0,0,0,17,17,104,109,250,51,
18,34,34,51,17,0,0,0,0,1,7,113,34,134,136,34,211,70,34,131,17,0,0,0,0,0,16,34,
34,134,34,34,97,17,102,98,16,0,0,0,0,0,1,18,85,84,17,98,97,33,22,98,1,0,0,0,0,1,
17,37,83,242,68,70,18,146,33,23,16,0,0,0,0,1,50,53,35,134,133,65,39,185,37,17,1,
0,0,0,0,0,19,147,34,136,51,81,39,190,36,65,32,0,0,0,0,0,17,18,34,131,34,83,39,0,
35,34,32,0,0,0,0,1,1,17,34,34,34,57,50,114,49,34,0,0,0,0,0,0,16,17,34,34,83,147,
67,33,64,0,0,0,0,0,0,0,1,2,33,35,81,17,68,68,65,0,0,0,0,0,0,0,0,2,33,21,17,17,
17,17,16,0,0,0,0,0,0,0,0,0,0,20,65,0,1,17,0,0,0,0,0,0,0,0,0,0,0,1,65,16,0,16,0,
0,0,0,0,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 3, dust
175,5,3,4,53,2,6,52,50,74,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,2,34,0,0,16,
1,0,0,0,0,0,0,0,0,0,0,34,50,32,3,48,0,0,0,0,0,0,0,0,0,0,2,35,50,16,66,18,0,0,0,
0,0,0,0,1,16,0,2,49,51,4,4,2,82,34,0,0,0,0,0,0,1,16,0,49,49,32,64,18,81,51,32,0,
0,0,0,0,0,0,0,3,22,55,97,18,33,19,48,0,0,0,0,0,0,0,0,0,6,19,50,34,19,17,48,0,0,
0,0,0,0,1,51,17,17,99,35,129,51,50,0,0,0,0,0,0,0,0,1,55,0,7,50,4,34,32,1,0,0,0,
0,0,0,0,3,112,64,2,64,32,98,0,0,0,0,0,0,0,0,0,9,52,4,3,4,4,5,0,0,0,0,0,0,0,0,0,
18,64,34,49,0,32,18,1,16,0,0,0,0,0,0,0,18,34,35,19,32,18,50,0,1,16,0,0,0,0,0,0,
0,37,19,34,136,18,51,16,0,0,0,0,0,0,0,0,0,5,35,49,51,1,0,16,0,0,0,0,0,0,0,0,0,1,
17,19,34,0,0,0,0,0,0,0,0,0,0,0,0,0,34,18,32,0,0,0,0,0,0,0,0,0,0,0,1,0,0,34,0,1,
0,0,0,0,0,0,0,0,0,0,16,0,16,0,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,0,0,16,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
};

SFG_PROGRAM_MEMORY uint8_t SFG_monsterSprites[19 * SFG_TEXTURE_STORE_SIZE] =
{
// 0, spider idle
175,0,4,3,5,6,223,1,7,10,2,20,62,95,120,18,0,1,17,16,0,0,0,0,0,0,0,0,1,17,16,0,
0,20,45,65,0,0,0,0,0,0,0,0,21,52,33,0,1,65,170,20,16,0,0,0,0,0,0,17,133,50,51,
16,1,218,211,162,16,0,0,0,0,0,17,133,84,17,17,17,1,42,61,173,16,1,0,0,0,1,133,
84,33,0,0,0,1,65,170,20,17,1,16,0,1,24,85,73,16,0,0,0,0,20,210,71,119,19,81,0,0,
1,19,146,16,0,0,0,1,115,17,115,115,68,65,0,0,0,1,66,49,0,0,0,20,36,119,41,37,69,
85,16,0,0,1,66,49,17,17,0,18,50,115,21,85,68,68,33,0,1,17,51,133,52,66,16,20,45,
113,28,197,66,66,65,1,24,136,85,85,50,35,49,1,17,37,86,198,196,34,33,0,1,85,85,
68,153,153,145,0,21,85,70,140,102,34,34,16,0,147,57,147,51,35,16,0,17,119,60,
104,102,98,34,46,238,191,66,243,17,17,17,1,57,115,115,38,108,198,34,231,243,187,
116,33,0,0,0,18,49,153,151,34,104,198,62,255,59,180,51,112,0,0,0,18,49,51,55,34,
104,198,62,255,59,180,51,112,0,0,0,1,57,121,115,38,108,198,34,231,243,187,116,
33,0,0,0,0,17,119,60,104,102,98,34,46,238,191,66,147,17,17,17,0,21,85,70,140,
102,34,34,16,0,148,41,147,51,35,16,1,17,37,86,198,196,34,33,0,1,136,136,133,153,
153,145,20,45,113,28,197,66,66,65,1,21,85,85,85,52,66,49,18,50,115,21,85,84,68,
33,0,1,17,51,133,50,35,16,20,36,119,41,37,85,85,16,0,0,1,66,55,17,17,0,1,115,17,
115,115,68,65,0,0,0,1,66,49,0,0,0,0,20,45,71,119,19,81,0,0,1,19,50,16,0,0,0,1,
65,170,20,17,1,16,0,1,24,136,137,16,0,0,0,1,218,61,162,16,1,0,0,0,1,133,85,129,
0,0,0,1,42,211,173,16,0,0,0,0,0,17,69,85,17,17,17,1,65,170,20,16,0,0,0,0,0,0,17,
69,50,35,16,0,20,210,65,0,0,0,0,0,0,0,0,20,50,49,0,0,1,17,16,0,0,0,0,0,0,0,0,1,
17,16,0
, // 1, spider attacking
175,0,183,3,4,1,191,5,31,6,7,22,10,223,20,18,0,34,37,85,43,176,0,0,0,0,0,0,17,
17,16,0,0,34,98,34,38,102,0,0,0,0,0,1,148,71,65,16,0,2,98,34,102,98,0,0,0,0,0,
89,169,52,51,49,0,2,104,40,102,32,0,0,0,0,5,170,145,17,17,16,2,38,104,136,98,32,
80,0,0,0,90,167,65,0,0,0,2,102,102,102,102,34,85,0,0,5,184,124,16,0,0,0,2,102,
34,34,102,102,232,80,0,91,139,76,16,0,0,0,2,34,46,34,38,102,187,113,0,85,84,196,
49,17,17,0,0,91,235,85,34,107,136,145,0,0,5,115,19,71,116,16,0,94,238,94,88,139,
139,116,80,0,85,58,169,52,67,49,0,23,75,85,91,184,231,71,80,5,170,153,119,60,
204,17,0,1,21,232,141,141,116,68,16,88,137,119,204,195,49,0,0,1,153,136,189,169,
212,68,65,1,195,60,51,51,67,16,0,1,17,85,55,218,221,212,68,85,95,116,243,49,17,
17,0,19,197,51,83,77,217,125,69,95,62,87,65,16,0,0,1,67,49,204,197,68,218,157,
95,243,235,51,80,0,0,0,1,67,193,51,53,68,218,157,95,243,235,51,80,0,0,0,0,19,
197,204,83,77,217,125,69,95,62,87,65,16,0,0,0,1,17,85,55,218,221,212,68,85,95,
116,243,49,17,17,0,1,153,153,125,168,221,68,65,5,51,60,51,51,67,16,0,1,17,232,
141,141,190,68,16,90,170,170,204,195,49,0,0,23,75,85,91,184,235,71,85,5,183,153,
170,60,204,17,0,94,238,94,88,184,187,116,80,0,85,55,153,52,67,49,2,43,235,82,34,
136,136,145,0,0,1,115,19,71,116,16,2,34,82,34,98,46,187,113,0,85,83,196,49,17,
17,0,2,98,34,38,102,44,232,16,0,91,170,124,16,0,0,0,2,102,102,102,98,37,85,0,0,
5,185,172,16,0,0,0,2,38,136,134,34,80,80,0,0,0,88,167,65,0,0,0,0,182,136,134,98,
80,0,0,0,0,5,185,145,17,17,16,0,34,102,102,102,32,0,0,0,0,0,87,153,52,51,49,0,2,
98,34,102,107,0,0,0,0,0,1,116,71,65,16,0,2,37,82,43,32,0,0,0,0,0,0,17,17,16,0
, // 2, spider walking
175,0,4,5,6,3,223,1,10,7,2,18,62,95,120,63,0,0,17,17,0,0,0,0,0,0,0,1,17,17,0,0,
0,1,50,211,16,0,0,0,0,0,0,21,51,50,16,0,0,19,26,161,49,0,0,0,0,0,17,146,35,33,0,
0,0,29,173,90,33,0,0,0,0,1,148,68,81,16,0,0,0,18,165,218,209,0,16,0,0,25,67,49,
16,0,0,0,0,19,26,161,49,16,17,0,1,148,51,16,0,0,0,0,0,1,61,35,119,113,84,16,25,
68,56,0,0,0,0,0,0,23,81,23,87,83,51,16,17,21,129,0,0,0,0,0,1,50,55,114,130,67,
68,65,0,1,50,16,0,0,0,0,1,37,39,81,68,67,51,50,0,1,85,81,17,17,17,0,1,50,215,17,
204,67,35,35,16,25,153,148,68,83,50,16,0,17,18,68,111,108,50,34,16,1,68,68,68,
82,37,81,0,1,68,67,105,246,98,34,33,0,133,88,136,136,136,129,0,1,23,117,198,150,
102,34,34,238,227,37,181,17,0,0,0,21,135,87,82,102,252,98,46,123,91,50,81,0,0,0,
1,37,24,136,114,38,159,101,235,181,35,181,112,0,0,0,1,37,24,136,114,38,159,101,
235,181,35,181,112,0,0,0,0,21,135,87,82,102,252,98,46,123,83,123,33,0,0,0,0,1,
23,117,198,150,102,34,34,238,235,50,133,17,17,0,0,1,68,67,105,246,98,34,33,0,
131,40,85,85,34,16,0,17,18,68,111,108,50,34,16,1,148,33,17,17,17,0,1,50,215,17,
204,67,35,35,16,25,50,16,0,0,0,0,1,37,39,81,68,67,51,50,17,148,72,136,128,0,0,0,
1,50,55,114,130,67,68,65,20,68,73,148,136,128,0,0,0,23,81,23,87,83,51,16,1,17,
84,68,83,56,129,0,0,1,61,35,119,113,84,16,0,5,21,68,82,34,81,0,0,19,26,161,49,
16,17,0,0,20,66,87,18,37,16,0,0,18,165,218,209,0,16,0,0,1,84,52,65,17,81,17,0,
29,173,90,33,0,0,0,0,0,17,19,52,82,37,16,0,19,26,161,49,0,0,0,0,0,0,1,19,82,81,
0,0,1,50,211,16,0,0,0,0,0,0,0,1,17,16,0,0,0,17,17,0,0,0,0,0,0,0,0,0,0,0,0
, // 3, destroyer idle
0,175,3,4,5,73,1,6,2,77,157,79,7,78,76,75,17,0,0,0,17,17,17,17,17,17,17,17,17,
17,17,17,17,11,221,219,0,17,17,17,16,0,0,0,0,0,1,17,17,11,217,153,219,1,17,16,2,
119,114,39,116,119,1,17,17,11,189,154,155,1,16,12,194,124,204,204,199,204,1,17,
17,16,189,154,169,176,12,199,116,35,51,51,50,51,1,17,17,17,11,217,170,213,196,
68,67,34,34,34,40,34,0,0,17,16,11,217,154,149,52,50,50,0,0,0,0,0,4,64,17,7,118,
102,138,173,83,68,128,17,17,17,17,7,114,32,16,116,12,66,106,169,181,51,71,0,1,
17,0,115,56,128,16,67,0,102,102,42,213,35,52,71,112,0,119,51,40,128,7,48,194,0,
7,114,171,82,51,68,71,114,83,34,40,128,4,12,204,64,0,71,106,181,131,52,68,71,82,
34,40,80,4,12,199,114,192,36,70,157,184,52,119,116,37,40,133,80,0,204,119,119,
96,4,70,169,213,55,51,55,69,80,102,96,0,102,103,116,44,4,50,106,155,83,157,147,
117,34,0,0,2,9,230,68,32,3,50,106,173,83,57,211,53,34,1,17,2,11,150,68,48,3,50,
106,233,83,57,211,53,34,1,17,0,34,36,67,44,3,50,106,238,83,157,147,37,34,128,0,
0,119,68,67,128,3,54,254,229,34,51,50,53,88,50,32,6,7,68,50,192,131,38,174,245,
35,34,35,37,34,40,128,2,4,68,48,0,50,111,229,82,51,51,50,82,34,40,128,3,32,66,0,
3,38,254,86,130,34,34,34,88,34,40,128,16,50,0,0,0,111,229,104,34,40,133,85,104,
136,40,80,16,51,99,38,105,154,245,130,40,96,82,67,6,102,133,80,17,3,54,102,137,
238,88,134,96,17,7,66,6,0,101,80,17,16,13,187,154,229,102,104,36,0,116,72,128,
17,6,96,17,17,11,186,170,245,96,130,51,71,67,34,32,17,16,0,17,16,187,218,174,
240,5,130,51,51,51,50,80,17,17,17,17,11,189,170,175,1,16,5,34,34,34,80,1,17,17,
17,17,11,221,170,255,1,17,16,85,85,0,1,17,17,17,17,17,11,218,255,0,17,17,17,0,0,
17,17,17,17,17,17,17,0,0,0,17,17,17,17,17,17,17,17,17,17,17,17
, // 4, destroyer attacking
175,0,3,4,73,1,5,2,183,6,77,157,79,78,7,31,0,0,0,0,0,0,0,0,0,0,128,0,0,0,0,0,0,
17,17,17,0,0,0,0,0,136,136,128,0,0,0,0,0,28,221,220,17,0,0,0,8,136,136,128,0,0,
0,0,0,28,218,170,220,16,1,88,136,136,255,136,128,0,0,0,0,28,218,187,172,85,94,
238,136,136,255,136,128,0,0,0,0,28,218,187,187,215,105,255,248,136,136,136,0,0,
0,0,0,1,221,171,187,163,102,255,255,136,136,136,0,0,1,17,0,25,17,85,87,189,38,
99,99,56,136,0,0,17,86,97,1,150,37,230,37,186,195,51,35,56,0,0,17,255,146,33,1,
99,37,85,85,82,211,35,54,98,17,17,249,99,55,113,25,51,94,33,17,153,44,50,51,111,
255,243,70,50,39,113,22,53,238,230,17,22,148,195,115,54,153,159,114,34,39,113,
19,49,238,153,46,18,102,125,151,57,51,57,100,34,39,65,19,94,233,153,149,17,102,
90,215,51,173,163,148,68,116,65,18,21,85,153,98,225,99,37,172,67,58,211,52,39,
69,81,19,33,163,86,98,17,51,37,189,67,58,211,52,34,17,17,18,33,202,86,99,17,51,
37,58,67,173,163,36,34,16,0,19,18,34,102,50,225,51,37,58,66,51,50,52,34,113,17,
18,89,150,102,55,17,51,82,52,35,34,35,36,71,50,33,19,33,150,99,46,23,50,82,36,
35,51,50,116,34,39,113,19,37,102,99,17,19,37,36,66,51,50,39,66,34,39,113,19,34,
22,33,17,50,82,69,114,34,36,68,71,34,39,113,1,50,81,17,85,85,36,87,34,39,66,99,
23,119,39,65,1,51,37,50,87,171,68,21,81,119,25,98,21,85,116,65,0,19,51,85,123,
50,65,87,38,17,150,103,113,17,84,65,0,1,92,202,171,52,81,35,51,105,99,34,33,0,
21,81,0,1,204,219,187,36,20,114,51,51,51,50,65,0,1,17,0,28,205,219,178,65,20,
119,34,34,34,65,16,0,0,0,0,28,205,187,36,16,1,20,68,68,65,16,0,0,0,0,0,28,219,
187,33,0,0,1,17,17,16,0,0,0,0,0,0,28,219,177,16,0,0,0,0,0,0,0,0,0,0,0,0,17,17,
16,0,0,0,0,0,0,0,0,0,0,0,0
, // 5, destroyer walking
175,0,3,4,1,5,2,73,6,77,79,7,78,157,75,76,0,17,17,17,0,0,0,0,0,0,0,0,0,0,0,0,0,
26,204,202,17,0,0,1,17,17,17,0,0,0,0,0,0,26,201,153,202,16,1,19,136,130,33,17,
17,0,0,0,0,26,172,157,154,16,27,179,40,187,184,133,129,16,0,0,0,1,172,157,217,
161,184,136,50,51,59,184,102,16,0,0,0,0,26,201,221,199,85,85,50,34,35,50,102,16,
0,0,0,1,26,201,157,151,53,35,97,17,18,38,102,16,17,16,0,24,17,68,70,220,115,54,
16,0,1,17,97,17,85,16,1,133,36,181,36,217,167,51,81,17,16,0,17,56,34,16,1,83,36,
68,68,119,199,35,53,88,129,17,136,51,102,16,24,51,75,33,17,136,122,114,51,85,88,
130,115,34,102,16,21,52,187,181,17,21,135,167,99,53,85,88,114,34,102,16,19,49,
187,136,43,18,85,76,214,53,136,133,39,34,103,16,19,75,184,136,132,17,85,73,199,
56,51,56,87,118,119,16,18,20,68,136,82,177,83,36,154,115,156,147,135,33,68,16,
19,33,159,69,82,17,51,36,220,115,57,195,55,34,17,16,18,33,169,69,83,17,51,36,
249,115,57,195,55,34,97,17,19,18,34,85,50,177,51,36,255,115,156,147,39,38,50,33,
18,72,133,85,54,17,51,78,247,34,51,50,55,114,38,97,19,33,133,83,43,22,50,78,231,
35,34,35,39,34,38,97,19,36,85,83,17,19,36,231,114,51,51,50,114,34,38,97,19,34,
21,33,17,50,78,116,98,34,34,38,118,34,38,113,1,50,65,17,68,68,231,70,34,38,97,
17,22,102,103,113,1,51,36,50,70,157,119,20,38,65,16,0,1,20,71,113,0,19,51,68,
109,254,113,70,65,23,97,16,0,1,20,65,0,1,74,169,157,247,65,35,37,136,133,33,0,0,
1,17,0,0,26,173,221,231,65,98,51,85,134,97,0,0,0,0,0,1,170,205,223,225,23,98,34,
51,51,33,0,0,0,0,0,26,172,221,222,16,1,17,118,34,38,97,0,0,0,0,0,26,204,221,238,
16,0,0,17,17,17,16,0,0,0,0,0,26,205,238,17,0,0,0,0,0,0,0,0,0,0,0,0,17,17,17,0,0,
0,0,0,0,0,0,0,0,0,0
, // 6, warrior idle
175,0,3,5,1,4,6,62,2,7,92,170,151,93,95,148,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,12,199,119,119,
119,204,204,187,176,0,0,0,0,0,0,0,185,153,153,153,153,150,103,119,123,0,0,0,0,0,
0,1,255,204,119,119,119,119,119,119,193,0,0,0,0,0,0,1,17,17,17,17,17,17,17,28,
133,16,0,0,0,0,0,0,0,0,0,0,0,0,0,1,85,16,0,0,1,17,16,0,0,0,0,0,0,0,0,0,21,81,0,
1,19,53,81,0,0,0,0,0,0,0,0,0,1,33,1,22,99,51,81,0,0,17,0,0,0,0,0,0,1,40,22,102,
51,51,33,0,0,25,16,0,17,16,0,0,18,40,153,102,99,50,68,0,0,22,145,17,234,145,0,1,
173,85,85,54,51,50,68,0,0,19,105,155,234,102,16,24,90,173,51,38,99,50,68,0,0,1,
51,54,178,54,129,130,55,218,211,38,51,50,68,0,0,0,21,51,102,99,136,34,55,122,
211,38,99,51,36,0,1,17,34,85,51,102,56,34,55,218,211,38,51,51,81,0,19,51,51,51,
51,51,88,34,58,173,50,86,99,53,81,0,1,17,34,37,51,53,88,34,170,51,35,102,50,34,
36,0,0,0,18,85,85,85,136,40,163,50,57,98,34,34,36,0,0,1,37,85,178,82,136,136,
162,43,149,34,180,68,68,0,0,21,85,43,234,34,65,132,162,43,82,36,68,34,36,0,0,21,
33,17,234,33,0,20,162,36,43,68,66,83,82,0,0,18,16,0,17,16,0,1,17,17,36,68,68,68,
68,0,0,17,0,0,0,0,0,0,0,1,43,68,68,34,36,0,0,0,0,0,0,0,0,0,0,0,18,43,66,83,82,0,
0,0,0,0,0,0,0,0,0,0,1,18,36,68,68,0,0,0,0,0,0,0,0,0,0,0,0,1,18,34,36,0,0,0,0,0,
0,0,0,0,0,0,0,0,1,17,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 7, warrior attacking
175,0,62,5,7,3,4,1,2,6,170,151,92,95,148,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
17,0,0,1,17,16,0,0,0,0,0,0,0,0,0,1,85,16,1,19,54,97,0,0,1,16,0,0,0,0,0,19,82,17,
25,147,51,97,0,0,1,145,16,1,17,0,1,53,126,74,89,51,51,87,0,0,1,153,17,29,201,16,
1,136,135,72,85,147,53,119,0,0,1,57,68,205,201,49,24,99,34,68,165,51,53,119,0,0,
0,19,51,154,163,152,136,50,36,68,133,83,53,119,0,0,0,1,86,57,153,150,133,50,34,
36,74,83,53,119,0,0,1,17,17,86,51,147,34,34,34,36,72,85,51,87,0,0,19,51,51,51,
51,54,133,34,43,180,68,165,51,97,0,0,1,17,17,83,54,102,133,51,34,34,68,133,83,
97,0,0,0,1,86,102,102,104,133,50,34,34,68,74,86,97,0,0,0,21,102,106,165,88,130,
34,68,68,68,69,133,87,0,0,1,86,101,205,197,135,120,34,34,34,36,66,167,119,0,0,1,
101,17,29,197,16,7,130,34,187,180,68,165,101,0,0,1,81,0,1,17,34,34,34,34,34,34,
68,42,119,0,0,1,16,0,0,0,2,34,36,68,68,68,68,74,85,0,0,0,0,0,0,0,0,0,34,34,68,
68,68,68,23,0,0,0,0,0,0,0,0,0,0,2,43,187,187,187,17,0,0,0,0,0,0,0,0,0,0,0,0,0,1,
17,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 8, plasmabot idle
175,0,72,65,2,3,4,5,61,60,54,62,6,59,49,66,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
2,34,34,0,0,0,0,0,0,0,0,0,0,0,0,0,42,170,170,34,0,0,0,0,0,0,1,17,0,0,0,0,42,147,
56,171,32,0,0,0,0,0,1,193,0,0,0,0,42,50,35,153,32,0,0,0,1,17,20,68,17,17,0,2,
171,50,34,57,32,0,0,0,1,119,118,102,85,65,0,2,163,34,34,57,33,0,0,0,1,101,85,68,
68,65,0,2,163,34,35,137,35,16,0,0,1,17,17,17,17,17,0,42,185,50,35,141,50,49,0,0,
0,0,1,65,0,0,0,42,187,131,57,211,66,52,16,0,0,0,21,81,0,0,0,43,51,136,179,68,
136,34,17,17,17,17,68,65,0,0,0,36,101,51,52,216,136,146,86,124,197,204,204,113,
0,0,0,3,122,229,61,136,137,242,17,17,17,17,28,17,0,0,0,2,122,174,83,153,211,35,
65,0,0,0,28,16,0,0,0,2,118,174,101,51,50,51,65,0,17,17,68,65,17,16,0,2,118,102,
238,85,82,52,65,1,119,119,102,102,85,65,0,2,118,102,238,102,98,52,81,1,102,85,
85,68,68,65,0,2,118,174,102,83,50,53,81,0,17,17,17,17,17,16,0,2,122,174,101,171,
179,37,81,0,0,0,20,16,0,0,0,3,122,230,93,184,136,210,17,17,17,17,21,17,0,0,0,35,
118,85,52,216,137,243,86,124,197,204,199,81,0,0,0,43,85,170,179,68,159,34,17,17,
17,17,68,65,0,0,0,42,171,147,57,211,66,52,16,0,0,0,23,113,0,0,0,42,185,50,35,
136,50,65,0,0,0,0,1,193,0,0,0,2,179,34,35,137,36,16,0,0,1,17,20,68,17,17,0,2,
179,34,34,57,33,0,0,0,1,119,118,102,85,65,0,2,152,50,34,61,32,0,0,0,1,101,85,68,
68,65,0,0,40,50,35,223,32,0,0,0,1,17,17,17,17,17,0,0,41,147,61,255,32,0,0,0,0,0,
1,81,0,0,0,0,47,255,223,34,0,0,0,0,0,0,1,17,0,0,0,0,2,34,34,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 9, plasmabot attacking
175,198,0,2,4,3,72,39,193,5,194,6,49,192,65,1,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,111,193,1,0,0,0,0,0,0,0,0,0,0,0,0,6,177,17,17,0,0,0,0,0,0,2,34,0,0,0,0,15,
113,17,17,240,0,0,0,0,0,2,178,0,0,0,0,107,17,23,17,31,0,0,0,8,136,131,51,34,34,
0,0,97,23,119,113,17,17,0,0,10,153,153,68,85,50,0,0,103,17,119,119,17,16,0,0,10,
68,85,83,51,50,0,6,145,17,119,17,19,61,0,0,10,168,141,34,34,34,0,1,17,17,23,17,
85,172,128,0,0,0,2,50,0,0,0,6,17,113,17,17,85,172,128,0,0,0,2,82,0,0,0,6,156,55,
113,85,84,76,200,170,168,136,211,50,0,0,0,6,52,85,81,84,180,68,245,73,187,91,
187,146,0,0,0,0,233,151,85,91,180,67,98,34,34,34,43,34,0,0,0,0,105,71,115,52,69,
230,227,32,0,0,43,32,0,0,0,0,105,68,124,83,206,110,227,32,168,141,51,50,34,32,0,
0,105,68,68,204,85,110,51,218,153,153,68,68,85,50,0,0,105,68,68,204,68,110,51,
218,68,85,85,51,51,50,0,0,105,68,124,69,238,254,53,208,136,136,221,210,34,32,0,
0,105,71,124,91,153,207,85,32,0,0,35,32,0,0,0,0,233,71,197,25,68,69,253,136,221,
221,37,34,0,0,0,6,233,149,81,20,148,67,197,73,187,91,185,82,0,0,0,6,149,87,17,
17,84,60,200,136,136,136,211,50,0,0,0,6,187,113,17,17,21,172,208,0,0,0,13,146,0,
0,0,6,177,17,119,17,17,163,32,0,0,0,13,178,0,0,0,6,73,145,23,119,19,50,0,0,10,
136,131,51,34,34,0,0,105,17,119,113,28,32,0,0,10,153,68,68,85,50,0,0,105,17,119,
17,17,0,0,0,10,69,85,51,51,50,0,0,97,17,23,17,22,0,0,0,10,168,210,34,34,34,0,0,
1,65,17,21,17,0,0,0,0,0,2,82,0,0,0,0,6,52,69,31,0,0,0,0,0,0,2,34,0,0,0,0,0,102,
102,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 10, ender idle
0,3,4,5,2,175,1,6,52,7,93,92,50,74,73,0,85,85,85,85,0,0,0,80,5,0,5,85,85,80,0,0,
85,85,80,0,153,153,52,1,134,17,16,80,0,9,114,32,85,80,3,151,119,119,119,104,97,
135,129,9,119,115,36,16,85,9,114,115,51,51,51,54,97,114,113,99,50,17,22,16,80,
151,113,18,4,18,34,51,97,135,129,102,33,20,70,16,9,119,20,55,114,0,65,34,16,17,
22,24,4,68,70,16,2,20,67,51,51,51,48,34,33,6,96,129,4,65,0,0,9,113,51,58,58,51,
115,3,49,65,51,0,65,96,85,85,9,49,33,34,35,34,35,49,51,18,34,38,102,102,0,0,7,
49,36,34,33,35,18,113,115,43,50,22,23,119,114,32,7,49,33,1,34,51,49,39,23,114,
51,22,18,34,36,16,7,19,34,64,66,34,51,19,35,51,50,6,102,17,22,96,80,115,35,11,
176,34,49,19,45,194,35,51,116,100,70,16,85,3,55,10,164,18,18,61,17,18,18,50,55,
97,22,16,80,1,121,10,170,65,35,18,34,34,33,34,35,96,0,0,3,68,121,64,0,1,18,34,
32,68,18,34,34,6,5,85,1,68,119,64,0,1,17,18,32,68,18,34,34,6,5,85,80,1,55,10,
170,65,34,65,34,34,33,34,33,96,0,0,85,3,51,10,164,35,34,52,17,18,17,34,36,103,
114,32,80,115,35,11,176,51,49,19,20,65,18,34,68,98,36,16,9,19,34,64,67,51,119,
19,35,50,33,70,6,17,22,96,7,49,33,1,51,51,113,35,19,52,34,16,100,68,70,16,3,49,
36,34,50,39,18,49,35,65,17,70,65,17,22,16,3,49,33,51,51,35,35,16,18,75,68,102,
102,102,0,0,3,33,35,51,51,51,49,1,17,68,70,0,98,96,85,85,1,68,18,58,58,49,208,
17,68,6,96,129,2,34,0,0,3,34,65,35,50,0,65,20,96,17,22,24,1,18,36,32,80,50,36,
29,4,18,17,70,97,135,129,0,17,17,20,16,85,2,36,34,34,33,20,230,97,114,113,100,
68,17,22,16,85,80,4,193,17,204,68,104,1,135,129,4,68,68,70,16,85,85,80,0,68,68,
70,1,128,17,16,80,0,4,70,16,85,85,85,85,0,0,0,80,5,0,5,85,85,80,0,0
, // 11, ender walking
0,3,4,5,175,2,1,6,52,93,7,92,50,74,73,0,68,68,68,68,64,0,0,4,0,64,0,68,68,0,0,4,
68,68,68,0,10,170,114,80,136,97,17,4,0,167,34,4,68,68,0,39,119,119,119,21,86,24,
120,16,119,50,81,4,68,0,167,35,51,51,51,49,102,23,39,22,33,17,97,4,64,42,114,18,
85,1,34,51,22,24,120,22,1,85,97,4,64,167,37,87,114,32,81,34,49,1,17,97,128,85,
97,4,2,21,85,51,51,51,53,18,34,16,102,8,16,81,0,4,10,119,19,51,147,147,115,1,49,
85,18,0,1,96,68,68,10,115,18,34,34,50,35,50,35,33,34,22,102,96,68,68,7,115,18,
81,34,18,49,55,19,50,178,22,23,114,0,0,7,115,18,16,18,35,50,23,23,115,35,22,18,
39,114,32,3,17,50,53,5,34,51,35,35,51,50,6,102,18,37,16,64,115,34,48,187,2,49,
19,45,194,35,51,117,101,22,96,64,50,39,112,153,177,18,49,209,17,33,35,55,97,86,
16,68,0,39,160,153,149,35,17,34,34,34,18,51,96,22,16,64,53,87,165,0,0,17,34,33,
5,82,34,35,6,0,0,64,21,87,165,0,0,17,18,33,5,82,34,34,6,4,68,68,0,39,112,153,
149,34,81,34,34,34,18,33,96,0,4,68,7,51,48,153,178,34,53,17,17,33,18,21,103,34,
4,64,115,35,48,187,3,49,19,21,81,18,17,85,98,81,4,10,17,34,37,5,51,55,19,34,51,
33,86,6,17,102,4,7,50,18,16,19,51,113,35,19,50,81,80,101,85,97,4,7,50,18,82,35,
39,18,49,83,37,21,86,81,17,97,4,3,50,18,35,51,50,35,16,18,37,181,102,102,96,0,4,
2,34,19,51,51,51,49,101,17,85,86,0,2,34,102,96,1,21,81,51,147,145,214,81,85,80,
102,1,128,18,37,32,64,50,37,19,50,37,1,21,86,1,17,104,16,17,21,16,64,51,33,93,
80,82,17,86,96,24,120,22,1,17,22,16,68,0,33,82,34,33,21,230,96,23,39,22,85,85,
86,16,68,68,0,92,17,204,85,96,16,24,120,16,85,85,86,16,68,68,68,0,5,85,86,0,136,
1,17,4,0,0,0,0,68,68,68,68,64,0,0,68,0,64,0,68,68,68,68,68
, // 12, ender attacking
0,4,5,175,3,26,28,191,6,29,2,1,30,7,105,27,51,51,51,51,48,0,0,3,0,48,112,51,51,
48,0,0,51,51,51,0,13,221,137,80,153,231,119,115,0,13,129,16,51,51,0,24,136,136,
136,101,94,119,221,119,136,130,26,64,51,0,216,18,204,204,204,198,119,125,221,
119,118,244,75,64,48,29,129,73,85,6,153,204,103,119,215,126,6,250,171,64,48,216,
26,168,137,144,86,153,198,7,119,118,144,90,171,64,1,74,170,34,34,34,37,105,153,
96,119,9,96,84,0,0,13,136,66,34,82,82,130,6,198,85,121,0,6,176,51,51,13,130,65,
17,17,33,18,41,156,150,153,110,187,187,0,0,8,130,65,164,17,65,36,40,108,201,89,
110,104,136,129,16,8,130,65,64,65,18,33,72,248,140,156,110,105,145,26,64,2,68,
33,42,170,17,34,18,18,34,41,14,235,255,75,176,48,130,17,42,85,1,36,66,26,81,18,
34,133,181,171,64,48,33,17,40,5,84,65,36,164,68,20,18,40,180,75,64,51,0,17,141,
5,90,18,68,17,17,17,65,34,176,0,0,48,42,161,141,0,0,68,17,20,10,161,17,18,11,3,
51,48,74,161,136,0,0,68,65,20,10,161,17,17,11,3,51,51,0,18,40,5,90,17,95,145,17,
17,65,20,176,0,0,51,8,34,33,5,81,17,37,255,255,31,65,74,184,129,16,48,130,18,42,
85,2,47,242,101,95,249,255,85,177,26,64,13,68,17,26,10,34,40,108,153,34,150,91,
11,244,75,176,8,33,65,64,66,34,134,156,108,201,86,80,181,170,171,64,8,33,65,161,
18,152,105,198,92,149,101,94,95,68,75,64,2,33,65,18,34,41,156,96,105,149,85,238,
235,187,0,0,1,17,66,34,34,44,198,229,102,85,126,0,9,176,51,51,4,74,164,34,82,
166,254,86,85,80,119,6,144,145,0,0,48,33,26,66,41,149,6,101,94,119,119,121,96,
97,26,16,48,34,20,95,80,89,102,94,224,119,215,126,6,111,74,64,51,0,20,89,153,
150,101,238,231,125,221,119,117,111,75,64,51,51,0,95,102,255,85,224,96,119,215,
112,85,90,171,64,51,51,51,0,5,85,94,0,153,7,119,3,0,10,171,64,51,51,51,51,48,0,
0,51,0,48,112,51,51,48,0,0
, // 13, turret idle
175,0,131,26,41,168,37,6,91,7,48,93,90,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,16,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,25,145,0,0,0,0,0,0,0,0,17,0,0,0,0,17,151,98,16,0,0,0,0,0,0,1,130,16,0,0,
1,35,184,130,81,16,0,0,0,0,0,1,40,16,0,0,1,34,123,178,85,65,0,0,0,0,0,23,49,0,0,
0,25,98,187,98,81,16,0,0,0,0,0,23,65,0,0,0,25,98,118,98,16,0,0,0,0,0,1,115,16,0,
17,17,71,98,118,98,81,16,0,0,0,1,20,116,16,0,0,0,22,35,114,36,85,65,0,0,1,25,
103,53,16,0,0,1,50,52,98,36,81,16,0,0,25,151,39,69,161,0,0,25,51,68,98,36,16,0,
0,0,25,118,100,84,161,0,1,151,54,35,184,36,16,0,0,1,151,98,35,52,85,17,23,102,
70,35,104,132,81,16,0,1,151,98,35,52,85,162,50,34,70,35,104,132,85,65,0,1,118,
34,51,68,85,17,19,51,70,35,104,132,81,16,0,1,118,35,51,68,81,0,1,67,70,35,184,
36,16,0,0,1,102,35,52,69,81,0,0,20,66,53,98,36,16,0,0,1,115,51,68,81,16,0,0,1,
67,165,98,36,81,16,0,1,114,51,69,16,0,0,0,0,20,85,98,36,85,65,0,27,98,35,65,0,0,
1,17,17,18,58,35,58,81,16,0,27,34,35,16,0,0,23,98,51,18,58,35,58,16,0,1,123,130,
52,16,0,0,22,34,68,18,58,140,58,81,16,23,98,184,129,0,0,0,1,35,65,1,90,44,202,
85,65,25,34,35,193,0,0,0,1,20,81,1,85,140,202,81,16,22,34,52,65,0,0,0,17,1,16,0,
17,35,58,16,0,1,35,68,16,0,0,0,16,0,0,0,0,19,49,0,0,0,17,17,0,0,0,0,0,0,0,0,0,1,
16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0
, // 14 turret walk
175,0,131,26,41,37,168,6,91,7,48,93,90,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,1,153,16,0,0,0,0,0,0,0,0,0,0,0,0,1,25,117,33,0,0,0,0,0,0,0,0,1,16,0,0,
18,59,136,33,0,0,0,0,0,0,0,0,24,33,0,0,18,39,187,38,16,0,0,0,0,0,0,0,18,129,0,1,
149,43,181,38,65,0,0,0,0,0,0,1,115,16,0,1,149,39,85,38,16,0,0,0,0,0,0,25,33,0,1,
20,117,39,85,33,0,0,0,0,0,1,17,114,49,0,16,1,82,55,34,70,16,0,0,0,1,25,87,51,16,
0,0,19,35,69,34,65,0,0,0,0,25,151,39,70,16,0,1,147,52,69,34,65,0,0,0,0,25,117,
84,100,161,0,25,115,82,59,130,70,16,0,0,1,151,82,35,52,161,17,117,83,82,53,136,
70,65,0,0,1,151,82,35,52,102,35,34,35,82,53,136,70,16,0,0,1,117,34,51,68,102,17,
51,52,82,53,136,65,0,0,0,1,117,34,51,68,102,0,20,52,82,59,130,65,0,0,0,23,85,35,
52,70,97,0,1,68,35,101,34,65,0,0,0,23,82,51,68,102,96,0,0,20,58,101,34,70,16,0,
1,181,34,52,65,16,0,17,16,1,70,101,34,65,0,0,23,178,34,52,16,0,1,119,81,17,35,
162,51,161,0,1,117,40,35,65,0,0,1,82,35,49,35,162,51,166,16,1,146,43,136,65,0,0,
17,34,52,17,35,168,195,166,65,1,82,34,60,16,0,0,1,51,65,0,22,162,204,166,16,1,
34,51,68,16,0,0,0,17,16,0,22,104,204,161,0,0,17,52,65,0,0,0,0,0,0,0,1,18,51,161,
0,0,0,17,16,0,0,0,0,0,0,0,0,1,51,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0
, // 15 turret attack
175,0,131,41,26,1,168,191,7,48,81,6,37,21,35,44,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,1,136,16,0,0,0,0,0,0,0,0,0,0,0,0,1,24,188,33,0,0,0,0,0,0,0,0,0,0,0,0,
18,77,170,38,16,0,0,0,0,0,0,0,0,0,0,0,18,43,212,38,49,0,0,0,0,0,0,0,0,0,0,1,140,
45,220,38,16,0,0,0,0,0,0,0,0,0,0,1,140,43,220,33,0,0,0,0,0,0,0,0,0,1,17,19,188,
43,220,38,16,0,0,0,0,80,85,0,0,0,0,1,194,75,226,54,49,0,0,0,85,135,188,165,0,0,
0,148,36,61,226,54,16,0,0,90,136,183,15,254,80,0,9,132,67,61,226,49,0,0,0,168,
139,119,112,238,80,0,24,180,194,77,162,49,0,0,1,187,183,120,135,116,161,21,189,
211,194,77,170,54,16,0,1,192,119,120,135,119,9,173,204,195,194,77,170,54,49,0,1,
204,192,119,119,106,169,21,221,211,194,77,170,54,16,0,0,175,255,247,122,170,80,
0,94,211,194,77,162,49,0,0,0,90,255,231,68,170,80,0,5,227,36,109,226,49,0,0,0,0,
85,224,74,85,0,0,0,83,73,109,226,54,16,0,0,0,0,85,85,0,0,0,0,1,54,109,226,54,49,
0,0,0,0,0,0,0,0,68,68,65,36,158,68,150,16,0,0,0,0,0,0,0,4,141,221,225,36,158,68,
145,0,0,0,0,0,0,0,0,4,221,35,49,36,154,164,150,16,0,0,0,0,0,0,0,0,18,67,16,22,
158,170,150,49,0,0,0,0,0,0,0,0,17,54,16,22,106,170,150,16,0,0,0,0,0,0,0,1,16,17,
0,1,30,68,145,0,0,0,0,0,0,0,0,1,0,0,0,0,1,68,16,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0
, // 16 exploder walk
175,168,1,2,3,102,4,5,23,174,99,191,6,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1,16,0,0,
0,0,89,0,0,0,0,0,0,0,0,68,65,177,0,0,5,85,89,0,0,0,0,0,0,0,4,34,33,177,0,0,19,
187,144,0,0,0,0,0,0,0,1,32,1,131,16,1,198,57,0,0,0,0,0,0,0,0,23,97,1,138,16,28,
116,16,0,0,0,0,0,0,17,17,34,34,17,138,17,17,65,0,0,0,0,0,0,0,22,50,51,51,50,134,
19,49,16,1,17,0,0,0,0,0,23,66,51,34,184,134,33,49,1,23,199,21,153,0,0,0,23,66,
34,184,133,85,50,17,28,199,103,99,85,80,0,0,23,66,136,138,18,85,106,50,199,119,
100,99,85,153,0,0,23,66,136,85,171,166,102,34,39,102,51,36,181,89,144,0,23,34,
34,165,90,21,130,33,36,35,55,103,75,89,0,1,18,187,136,136,81,168,36,116,18,119,
118,70,59,185,144,24,139,85,85,102,98,88,39,71,18,102,102,68,59,187,153,1,18,51,
170,170,97,168,36,116,18,68,68,68,59,185,144,0,19,34,34,58,106,21,130,33,39,34,
51,68,75,89,0,0,23,50,35,35,171,166,163,34,54,103,51,36,85,153,0,0,23,66,42,51,
18,102,163,34,52,70,102,115,89,144,0,0,23,66,34,163,51,86,50,17,19,52,68,97,89,
0,0,0,23,66,50,34,58,90,33,49,1,19,51,16,0,0,0,0,22,50,51,51,34,90,19,49,16,1,
17,0,0,0,0,0,17,17,34,34,17,138,17,17,193,0,0,0,0,0,0,0,0,0,20,65,1,131,16,20,
124,16,0,0,0,0,0,0,0,0,1,32,1,131,16,1,70,57,0,0,0,0,0,0,0,0,1,18,33,129,0,0,19,
187,144,0,0,0,0,0,0,0,0,17,17,129,0,0,5,155,153,0,0,0,0,0,0,0,0,0,1,16,0,0,0,5,
80,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0
, // 17 universal dying sprite
175,0,223,1,3,4,5,2,6,73,135,79,26,159,131,37,0,0,0,0,0,0,0,0,0,0,0,51,0,0,0,0,
0,0,0,0,0,2,0,0,0,34,32,51,0,0,0,0,0,0,0,0,0,2,32,0,2,163,34,0,0,0,0,0,0,0,0,0,
0,0,0,0,42,3,50,33,16,0,0,0,0,0,0,0,0,0,1,16,32,0,2,22,97,16,0,0,0,0,0,0,0,1,24,
129,16,0,1,119,85,81,0,0,0,0,0,0,0,24,134,134,1,16,1,68,119,84,16,0,0,0,0,0,0,
24,102,136,102,81,0,20,68,116,16,0,0,0,0,0,1,102,102,104,134,65,2,20,68,151,113,
0,0,0,0,0,1,101,86,102,101,48,2,1,68,73,113,0,0,0,0,0,1,101,85,102,100,48,35,52,
17,73,16,0,0,0,0,0,1,85,85,85,71,51,37,84,48,17,0,0,0,0,0,0,1,84,80,80,71,50,34,
36,48,0,0,0,0,0,0,0,0,20,68,68,115,34,85,34,35,0,51,0,0,0,0,0,0,1,68,19,50,42,
51,85,34,3,48,0,0,0,0,0,0,0,17,34,34,163,101,51,67,32,0,0,0,0,0,0,0,0,2,42,68,
83,67,85,49,0,0,0,0,0,0,0,0,2,34,0,53,51,53,51,84,16,0,0,0,0,0,0,0,0,32,0,1,187,
211,115,21,16,0,0,0,0,0,0,0,2,32,0,27,189,211,48,23,16,0,0,0,0,0,0,0,2,0,0,27,
221,209,1,97,0,0,0,0,0,0,0,0,0,0,1,155,221,65,1,112,1,17,16,0,0,0,0,0,0,0,22,
153,145,17,22,49,22,117,16,0,0,0,0,0,0,1,101,84,113,0,23,54,102,53,16,0,0,0,0,0,
0,33,85,84,121,16,1,85,68,55,81,0,0,0,0,0,2,33,68,73,17,193,1,68,71,115,65,0,0,
0,0,0,34,42,153,145,1,195,16,23,119,115,65,0,0,0,0,0,32,2,170,34,31,236,49,23,
116,17,17,0,0,0,0,0,0,0,34,2,24,238,193,1,17,0,0,0,0,0,0,0,0,0,32,0,24,254,193,
0,35,48,0,0,0,0,0,0,0,0,0,0,1,17,16,2,32,48,0,0,0,0,0,0,0,0,0,0,0,1,0,2,0,0,0
, // 18 universal dead/corpse sprite
175,0,3,4,1,5,6,2,157,18,95,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0,0,0,4,68,0,0,0,0,0,0,0,0,0,0,0,0,0,0,66,34,0,0,0,0,0,0,0,0,0,0,0,
0,0,4,50,34,0,0,0,0,0,0,0,0,0,0,0,0,0,4,36,17,0,0,0,0,0,0,0,0,0,0,0,0,1,18,65,
49,0,0,0,0,0,0,0,0,0,0,1,17,34,34,19,51,0,0,0,0,0,0,0,0,0,1,22,98,34,34,33,20,0,
0,0,0,0,0,0,0,0,22,101,101,40,130,36,68,0,0,0,0,0,0,0,0,0,22,85,102,85,136,129,
17,0,0,0,0,0,0,0,0,1,85,85,82,37,136,22,68,0,0,0,0,0,0,0,0,1,18,53,85,34,132,19,
36,0,0,0,0,0,0,0,0,0,2,34,85,84,33,100,66,0,0,0,0,0,0,0,0,1,35,51,51,39,22,102,
49,0,0,0,0,0,0,0,0,1,50,58,58,39,65,101,33,0,0,0,0,0,0,0,0,0,18,34,34,116,22,83,
17,0,0,0,0,0,0,0,0,0,1,34,20,68,17,50,20,0,0,0,0,0,0,0,0,0,0,17,4,68,34,20,52,0,
0,0,0,0,0,0,0,0,0,0,0,34,39,83,49,0,0,0,0,0,0,0,0,0,0,0,34,35,117,51,65,0,0,0,0,
0,0,0,0,0,0,0,0,19,115,51,52,0,0,0,0,0,0,0,0,0,0,0,0,19,55,163,153,0,0,0,0,0,0,
0,0,0,0,0,0,1,55,51,153,0,0,0,0,0,0,0,0,0,0,0,0,0,17,115,41,0,0,0,0,0,0,0,0,0,0,
0,0,17,53,36,65,0,0,0,0,0,0,0,0,0,0,0,0,21,33,17,18,0,0,0,0,0,0,0,0,0,0,0,0,1,
16,1,35,0,0,0,0,0,0,0,0,0,0,0,0,0,0,18,50,0,0,0,0,0,0,0,0,0,0,0,0,0,0,17,34,0,0,
0,0,0,0,0,0,0,0,0,0,0,0,2,17,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,34,0,0,0,0,0,0,0,0,0,
0,0,0,0,0,0,0
};

SFG_PROGRAM_MEMORY uint8_t SFG_logoImage[SFG_TEXTURE_STORE_SIZE] =
{
175,224,0,7,4,5,162,77,2,74,70,69,6,11,68,75,0,0,0,0,0,0,17,17,17,17,17,0,0,0,0,
0,0,0,0,0,0,1,132,69,51,84,65,0,0,0,0,0,0,0,0,0,0,24,65,17,19,17,17,0,0,0,0,0,0,
0,0,0,0,24,65,0,19,16,0,0,0,0,0,0,0,0,0,0,0,24,65,17,19,17,17,0,0,0,0,0,0,0,0,0,
0,1,132,69,51,84,65,32,0,0,0,0,0,0,0,0,0,1,17,17,17,17,17,242,0,0,0,0,0,0,0,6,
97,243,94,40,51,84,66,159,32,0,0,0,0,0,17,17,24,60,190,40,50,34,34,155,146,0,0,
0,0,0,33,17,20,203,238,109,51,84,66,251,178,0,0,0,0,0,17,17,131,171,123,145,18,
34,38,122,187,32,0,0,0,0,6,97,92,170,167,113,115,35,81,122,171,32,0,0,0,0,2,22,
202,170,170,212,51,37,210,26,121,130,0,0,0,0,34,17,22,234,164,28,60,34,92,23,
218,56,32,0,0,2,33,17,102,110,113,69,210,35,51,93,51,66,50,0,0,34,34,40,126,246,
210,34,195,35,51,49,210,34,34,0,0,0,2,40,119,238,105,197,210,35,51,93,51,65,50,
0,0,0,2,38,87,119,166,156,60,34,92,223,218,161,32,0,0,2,34,38,87,119,170,20,51,
37,223,23,114,18,0,0,0,2,34,34,69,122,119,241,211,35,86,102,187,32,0,0,0,0,0,97,
213,119,187,97,134,45,17,246,105,32,0,0,0,0,34,33,22,247,238,29,51,53,65,155,98,
0,0,0,0,0,2,34,33,110,233,29,54,17,17,155,146,0,0,0,0,0,0,0,97,111,153,34,34,34,
29,159,32,0,0,0,0,0,0,0,0,41,150,40,51,84,65,146,0,0,0,0,0,0,0,0,0,2,38,40,56,
24,65,32,0,0,0,0,0,0,0,0,0,0,2,40,49,145,65,0,0,0,0,0,0,0,0,0,0,1,17,34,17,17,
17,0,0,0,0,0,0,0,0,0,0,1,132,69,51,84,65,0,0,0,0,0,0,0,0,0,0,1,17,24,56,17,17,0,
0,0,0,0,0,0,0,0,0,0,0,24,51,84,65,0,0,0,0,0,0,0,0,0,0,0,0,1,17,17,17,0,0,0,0,0
};

uint8_t SFG_charToFontIndex(char c)
{
  if (c >= 'a' && c <= 'z')
    return c - 'a';

  if (c >= 'A' && c <= 'Z')
    return c - 'A';

  if (c >= '0' && c <= '9')
    return c - '0' + 31;

  switch (c)
  {
    case ' ': return 26; break;
    case '.': return 27; break;
    case ',': return 28; break;
    case '!': return 29; break;
    case '/': return 41; break;
    case '-': return 42; break;
    case '+': return 43; break;
    case '(': return 44; break;
    case ')': return 45; break;
    case '%': return 46; break;
    default:  return 30; break; // "?"
  }
}

#define SFG_FONT_CHARACTER_SIZE 4 ///< width (= height) of font char. in pixels

/**
  4x4 font, each character stored as 16 bits.
*/
static const uint16_t SFG_font[47] =
{
  0xfaf0, // 0 "A"
  0xfd70, // 1 "B"
  0x6990, // 2 "C"
  0xf960, // 3 "D"
  0xfd90, // 4 "E"
  0xfa80, // 5 "F"
  0x69b0, // 6 "G"
  0xf4f0, // 7 "H"
  0x9f90, // 8 "I"
  0x31f0, // 9 "J"
  0xf4b0, // 10 "K"
  0xf110, // 11 "L"
  0xfc4f, // 12 "M"
  0xf42f, // 13 "N"
  0x6996, // 14 "O"
  0xfae0, // 15 "P"
  0x69b7, // 16 "Q"
  0xfad0, // 17 "R"
  0x5da0, // 18 "S"
  0x8f80, // 19 "T"
  0xf1f0, // 20 "U"
  0xe1e0, // 21 "V"
  0xf32f, // 22 "W"
  0x9690, // 23 "X"
  0xc7c0, // 24 "Y"
  0xbd90, // 25 "Z"
  0x0000, // 26 " "
  0x0100, // 27 "."
  0x0300, // 28 ","
  0x0d00, // 29 "!"
  0x48b4, // 30 "?"
  0xf9f0, // 31 "0"
  0x9f10, // 32 "1"
  0xbdd0, // 33 "2"
  0x9da0, // 34 "3"
  0xe2f0, // 35 "4"
  0xdbb0, // 36 "5"
  0xfbb0, // 37 "6"
  0x8bc0, // 38 "7"
  0xfdf0, // 39 "8"
  0xddf0, // 40 "9"
  0x1680, // 41 "/"
  0x2220, // 42 "-"
  0x2720, // 43 "+"
  0x0690, // 44 "("
  0x0960, // 45 ")"
  0x9249  // 46 "%"
};

#endif // guard
